<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Arcdeco Hero Slider Widget.
 *
 * @since 1.0
 */
class Arcdeco_Hero_Slider_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-hero-slider';
	}

	public function get_title() {
		return esc_html__( 'Hero Slider', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fa fa-tv';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'arcdeco-plugin' ),
				'default' => esc_html__( 'Enter title', 'arcdeco-plugin' ),
			]
		);

		$repeater->add_control(
			'description', [
				'label'       => esc_html__( 'Description', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'arcdeco-plugin' ),
				'default' => esc_html__( 'Enter description', 'arcdeco-plugin' ),
			]
		);

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Image', 'arcdeco-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'more_link', [
				'label' => esc_html__( 'URL', 'arcdeco-plugin' ),
				'label_block' => true,
				'type' => Controls_Manager::URL,
				'show_external' => true,
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'arcdeco-plugin' ),
				'type' => Controls_Manager::REPEATER,  
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'social_tab',
			[
				'label' => esc_html__( 'Social', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);	

		$this->add_control(
			'social',
			[
				'label' => esc_html__( 'Show Social', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'arcdeco-plugin' ),
				'label_off' => esc_html__( 'Hide', 'arcdeco-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'slider_settings_tab',
			[
				'label' => esc_html__( 'Slider Settings', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'slider_autoplay',
			[
				'label' => esc_html__( 'Autoplay', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'arcdeco-plugin' ),
				'label_off' => esc_html__( 'No', 'arcdeco-plugin' ),
				'return_value' => 1,
				'default' => 1,
			]
		);

		$this->add_control(
			'slider_delay',
			[
				'label' => esc_html__( 'Delay', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'ms' ],
				'range' => [
					'ms' => [
						'min' => 1000,
						'max' => 100000,
						'step' => 10,
					],
				],
				'default' => [
					'unit' => 'ms',
					'size' => 2500,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label' => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .hero .slider__content .title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .hero .slider__content .title--separation::before' => 'background: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .hero .slider__content .title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'description_styling',
			[
				'label' => esc_html__( 'Description', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .hero .slider__content .description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .hero .slider__content .description',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$social_links = get_field( 'social_links', 'option' );

		?>

		<!-- Slider -->
	    <main class="hero slider js-scroll">
			<div class="slider__caption swiper-container">
	            <div class="swiper-wrapper">
				    <?php if ( $settings['items'] ) : ?>
				    <?php 
				        foreach ( $settings['items'] as $index => $item ) : 
				        $item_title = $this->get_repeater_setting_key( 'title', 'items', $index );
				    	$this->add_inline_editing_attributes( $item_title, 'basic' );

				    	$item_description = $this->get_repeater_setting_key( 'description', 'items', $index );
				    	$this->add_inline_editing_attributes( $item_description, 'basic' );
				    ?>
				    <!-- Caption -->
				    <div class="swiper-slide">
						<div class="slider__content" data-swiper-parallax="-200" data-swiper-parallax-opacity="0">
						    <?php if ( $item['more_link'] ) : ?>
						    <a<?php if ( $item['more_link']['is_external'] ) : ?> target="_blank"<?php endif; ?><?php if ( $item['more_link']['nofollow'] ) : ?> rel="nofollow"<?php endif; ?> href="<?php echo esc_url( $item['more_link']['url'] ); ?>">
							<?php endif; ?>
						    	<?php if ( $item['title'] ) : ?>
							    <h1 class="title title--display-1 title--separation">
							    	<span <?php echo $this->get_render_attribute_string( $item_title ); ?>>
						    			<?php echo wp_kses_post( $item['title'] ); ?>
						    		</span>
							    </h1>
								<?php endif; ?>
								<?php if ( $item['description'] ) : ?>
							    <p class="description">
							    	<span <?php echo $this->get_render_attribute_string( $item_description ); ?>>
						    			<?php echo wp_kses_post( $item['description'] ); ?>
						    		</span>
							    </p>
							    <?php endif; ?>
							<?php if ( $item['more_link'] ) : ?>
							</a>
							<?php endif; ?>
						</div>
					</div>
				    <!-- /Caption -->
				    <?php endforeach; ?>
				    <?php endif; ?>
			    </div>
			</div>	
			
			<div class="slider__image swiper-container overlay--45"<?php if ( $settings['slider_delay'] ) : ?> data-swiper-delay="<?php echo esc_attr( $settings['slider_delay']['size'] ); ?>"<?php endif; ?> data-swiper-autoplay="<?php echo (int) esc_attr( $settings['slider_autoplay'] ); ?>">
	            <div class="swiper-wrapper">
				    <?php if ( $settings['items'] ) : ?>
				    <?php foreach ( $settings['items'] as $item ) :
				    if ( $item['image'] ) :
				    $image = wp_get_attachment_image_url( $item['image']['id'], 'arcdeco_1920xAuto' );
				    ?>
				    <!-- Image -->
				    <div class="swiper-slide">
						<div class="scaleCover"><div class="cover-slider js-image" data-image="<?php echo esc_url( $image ); ?>"></div></div>
					</div>
				    <!-- /Image -->
					<?php endif; ?>
				    <?php endforeach; ?>
				    <?php endif; ?>
			    </div>
			</div>
				
			<!-- Navigation buttons -->
			<div class="slider-navigation">
	            <div class="slider-prev zoom-cursor magnetic">
					<svg width="15" height="25" viewBox="0 0 15 25" fill="none" xmlns="http://www.w3.org/2000/svg">
	                    <path fill-rule="evenodd" clip-rule="evenodd" d="M14.5001 1.95233L12.5413 0L0 12.5L12.5413 25L14.374 23.1732L3.79156 12.6256L14.5001 1.95233Z" fill="#C4C4C4"/>
	                </svg>
				</div>
	            <div class="slider-next zoom-cursor magnetic">
					<svg width="15" height="25" viewBox="0 0 15 25" fill="none" xmlns="http://www.w3.org/2000/svg">
	                    <path fill-rule="evenodd" clip-rule="evenodd" d="M0.125977 1.827L10.708 12.3744L0 23.0474L1.95898 25L14.5 12.5L1.95898 0L0.125977 1.827Z" fill="#C4C4C4"/>
	                </svg>
				</div>
			</div>
		
			<!-- Control -->
			<div class="slider__control">
				<div class="slider-pagination-progressbar"></div>
				<div class="slider-pagination-fraction"></div>
			</div>
			
			<?php if ( $settings['social'] == 'yes' && $social_links ) : ?>
			<!-- Social -->
			<div class="social social-floating">
				<?php foreach ( $social_links as $social_link ) : ?>
		        <a class="social__link magnetic" href="<?php echo esc_url( $social_link['url'] ); ?>"><i class="<?php echo esc_attr( $social_link['icon'] ); ?>"></i></a>
		    	<?php endforeach; ?>
		    </div>
		    <?php endif; ?>
	    </main>
	    <!-- /Slider -->
	    
		<?php 
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() { ?>
		
		<!-- Slider -->
	    <main class="hero slider js-scroll">
			<div class="slider__caption swiper-container">
	            <div class="swiper-wrapper">
				    <# if ( settings.items ) { #>
				    <# _.each( settings.items, function( item, index ) {
				        var item_title = view.getRepeaterSettingKey( 'title', 'items', index );
						view.addInlineEditingAttributes( item_title, 'basic' );

						var item_description = view.getRepeaterSettingKey( 'description', 'items', index );
						view.addInlineEditingAttributes( item_description, 'basic' );
					#>
				    <!-- Caption -->
				    <div class="swiper-slide">
						<div class="slider__content" data-swiper-parallax="-200" data-swiper-parallax-opacity="0">
						    <# if ( item.more_link ) { #>
						    <a<# if ( item.more_link.is_external ) { #> target="_blank"<# } #><# if ( item.more_link.nofollow ) { #> rel="nofollow"<# } #> href="{{{ item.more_link.url }}}">
						    <# } #>
						    	<# if ( item.title ) { #>
							    <h1 class="title title--display-1 title--separation">
							    	<span {{{ view.getRenderAttributeString( item_title ) }}}>{{{ item.title }}}</span>
							    </h1>
								<# } #>
								<# if ( item.description ) { #>
							    <p class="description">
							    	<span {{{ view.getRenderAttributeString( item_description ) }}}>{{{ item.description }}}</span>
							    </p>
							    <# } #>
							<# if ( item.more_link ) { #>
							</a>
							<# } #>
						</div>
					</div>
				    <!-- /Caption -->
				    <# }); #>
				    <# } #>
			    </div>
			</div>	
			
			<div class="slider__image swiper-container overlay--45"<# if ( settings.slider_delay ) { #> data-swiper-delay="{{{ settings.slider_delay.size }}}"<# } #> data-swiper-autoplay="{{{ settings.slider_autoplay }}}">
	            <div class="swiper-wrapper">
				    <# if ( settings.items ) { #>
				    <# _.each( settings.items, function( item ) { #>
				    <!-- Image -->
				    <div class="swiper-slide">
						<div class="scaleCover"><div class="cover-slider js-image" data-image="{{{ item.image.url }}}"></div></div>
					</div>
				    <!-- /Image -->
				    <# }); #>
				    <# } #>
			    </div>
			</div>
				
			<!-- Navigation buttons -->
			<div class="slider-navigation">
	            <div class="slider-prev zoom-cursor magnetic">
					<svg width="15" height="25" viewBox="0 0 15 25" fill="none" xmlns="http://www.w3.org/2000/svg">
	                    <path fill-rule="evenodd" clip-rule="evenodd" d="M14.5001 1.95233L12.5413 0L0 12.5L12.5413 25L14.374 23.1732L3.79156 12.6256L14.5001 1.95233Z" fill="#C4C4C4"/>
	                </svg>
				</div>
	            <div class="slider-next zoom-cursor magnetic">
					<svg width="15" height="25" viewBox="0 0 15 25" fill="none" xmlns="http://www.w3.org/2000/svg">
	                    <path fill-rule="evenodd" clip-rule="evenodd" d="M0.125977 1.827L10.708 12.3744L0 23.0474L1.95898 25L14.5 12.5L1.95898 0L0.125977 1.827Z" fill="#C4C4C4"/>
	                </svg>
				</div>
			</div>
		
			<!-- Control -->
			<div class="slider__control">
				<div class="slider-pagination-progressbar"></div>
				<div class="slider-pagination-fraction"></div>
			</div>
	    </main>
	    <!-- /Slider -->

	<?php }
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Hero_Slider_Widget() );