<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Arcdeco Portfolio Widget.
 *
 * @since 1.0
 */
class Arcdeco_Portfolio_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-portfolio';
	}

	public function get_title() {
		return esc_html__( 'Portfolio', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fas fa-suitcase';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'heading_tab',
			[
				'label' => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Title', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => [
					'h1'  => __( 'H1', 'arcdeco-plugin' ),
					'h2' => __( 'H2', 'arcdeco-plugin' ),
					'h3' => __( 'H3', 'arcdeco-plugin' ),
					'div' => __( 'DIV', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'title_show',
			[
				'label' => esc_html__( 'Show Title', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'arcdeco-plugin' ),
				'label_off' => __( 'Hide', 'arcdeco-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'filters_tab',
			[
				'label' => esc_html__( 'Filters', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'filters',
			[
				'label' => esc_html__( 'Show Filters', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'arcdeco-plugin' ),
				'label_off' => __( 'Hide', 'arcdeco-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'source',
			[
				'label'       => esc_html__( 'Source', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all'  => __( 'All', 'arcdeco-plugin' ),
					'categories' => __( 'Categories', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'source_categories',
			[
				'label'       => esc_html__( 'Source', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_portfolio_categories(),
				'condition' => [
		            'source' => 'categories'
		        ],
			]
		);

		$this->add_control(
			'limit',
			[
				'label'       => esc_html__( 'Number of Items', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => 8,
				'default'     => 8,
			]
		);

		$this->add_control(
			'sort',
			[
				'label'       => esc_html__( 'Sorting By', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'menu_order',
				'options' => [
					'date'  => __( 'Date', 'arcdeco-plugin' ),
					'title' => __( 'Title', 'arcdeco-plugin' ),
					'rand' => __( 'Random', 'arcdeco-plugin' ),
					'menu_order' => __( 'Order', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'       => esc_html__( 'Order', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'asc',
				'options' => [
					'asc'  => __( 'ASC', 'arcdeco-plugin' ),
					'desc' => __( 'DESC', 'arcdeco-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'settings_tab',
			[
				'label' => esc_html__( 'Settings', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'       => esc_html__( 'Pagination', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'no',
				'options' => [
					'no'  => __( 'No', 'arcdeco-plugin' ),
					'scroll' => __( 'Infinite Scrolling', 'arcdeco-plugin' ),
					'pages' => __( 'Pages', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'columns',
			[
				'label'       => esc_html__( 'Columns', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => '3',
				'options' => [
					'3'  => __( '3 Columns', 'arcdeco-plugin' ),
					'2' => __( '2 Columns', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'item_image',
			[
				'label'       => esc_html__( 'Image Orientation', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1'  => __( 'Default', 'arcdeco-plugin' ),
					'2' => __( 'Vertical', 'arcdeco-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .title--h1' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .title--h1',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'filters_styling',
			[
				'label'     => esc_html__( 'Filters', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'filters_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .filter__link' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'filters_typography',
				'selector' => '{{WRAPPER}} .filter__link',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .content-grid .title' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .content-grid .title',
			]
		);

		$this->add_control(
			'item_text_color',
			[
				'label'     => esc_html__( 'Description Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .content-grid__description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_text_typography',
				'label'     => esc_html__( 'Description Typography', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .content-grid__description',
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'pagination_styling',
			[
				'label'     => esc_html__( 'Pagination', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'pagination_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .page-numbers' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'pagination_typography',
				'selector' => '{{WRAPPER}} .page-numbers',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render Categories List.
	 *
	 * @since 1.0
	 */
	protected function get_portfolio_categories() {
		$categories = [];

		$args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'portfolio_categories',
			'pad_counts'	=> false 
		);

		$portfolio_categories = get_categories( $args );

		foreach ( $portfolio_categories as $category ) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		
		$this->add_inline_editing_attributes( 'title', 'basic' );

		$paged = ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1;
		$page_id = get_the_ID();

		if ( $settings['source'] == 'all' ) {
			$cat_ids = '';
		} else {
			$cat_ids = $settings['source_categories'];
		}

		$cat_args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'portfolio_categories',
			'pad_counts'	=> false,
			'include'		=> $cat_ids
		);

		$pf_categories = get_categories( $cat_args );

		$args = array(
			'post_type'			=> 'portfolio',
			'post_status'		=> 'publish',
			'orderby'			=> $settings['sort'],
			'order'				=> $settings['order'],
			'posts_per_page'	=> $settings['limit'],
			'paged' 			=> $paged
		);

		if( $settings['source'] == 'categories' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'portfolio_categories',
					'field'    => 'id',
					'terms'    => $cat_ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}

		$q = new \WP_Query( $args );

		?>

		<div class="container">
		    <header class="header-page">
		    	<?php if ( $settings['title'] && $settings['title_show'] ) : ?>
		        <<?php echo esc_attr( $settings['title_tag'] ); ?> class="title title--h1 js-lines">
		        	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>><?php echo wp_kses_post( $settings['title'] ); ?></span>
		        </<?php echo esc_attr( $settings['title_tag'] ); ?>>
		        <?php endif; ?>

		        <?php if ( $settings['filters'] && $pf_categories ) : ?>
		        <!-- Filter -->
		        <div class="select js-down-done">
			        <span class="placeholder"><?php echo esc_html__( 'Select category', 'arcdeco-plugin' ); ?></span>
			        <ul class="filter">
			            <li class="filter__item"><?php echo esc_html__( 'Category', 'arcdeco-plugin' ); ?></li>
				        <li class="filter__item active" data-filter="*"><a class="filter__link active" href="#filter"><?php echo esc_html__( 'All', 'arcdeco-plugin' ); ?></a></li>
				        <?php foreach ( $pf_categories as $category ) : ?>
				        <li class="filter__item" data-filter=".category-<?php echo esc_attr( $category->slug ); ?>"><a class="filter__link" href="#filter"><?php echo esc_html( $category->name ); ?></a></li>
				        <?php endforeach; ?>
			        </ul>
			        <input type="hidden" name="changemetoo"/>
		        </div>
		        <?php endif; ?>
		    </header>

		    <?php if ( $q->have_posts() ) : ?>
		    <div class="content-grid projects-grid<?php if ( $settings['item_image'] == '1' ) : ?> project-grid<?php endif; ?> project-masonry filter-container<?php if ( $settings['pagination'] == 'scroll' ) : ?> portfolio-infinite-scrolling<?php endif; ?><?php if ( $settings['columns'] == '2' ) : ?> portfolio-columns-2<?php endif; ?>">
		        <div class="gutter-sizer"></div>

		        <?php while ( $q->have_posts() ) : $q->the_post();
					get_template_part( 'template-parts/content', get_post_type() );
				endwhile; ?>			
		    </div>
		    <?php if ( $settings['pagination'] == 'pages' ) : ?>
		    	<div class="pager">
		    		<?php
					$big = 999999999; // need an unlikely integer

					echo paginate_links( array(
						'base' => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
						'format' => '?paged=%#%',
						'current' => max( 1, get_query_var('paged') ),
						'total' => $q->max_num_pages,
						'prev_text' => esc_html__( 'Prev', 'arcdeco-plugin' ),
						'next_text' => esc_html__( 'Next', 'arcdeco-plugin' ),
					) ); ?>
				</div>
				<?php endif;
				if ( $settings['pagination'] == 'scroll' ) :
					$infinite_scrolling_data = array(
						'url'   => admin_url( 'admin-ajax.php' ),
						'max_num' => $q->max_num_pages,
						'page_id' => $page_id,
						'order_by' => $settings['sort'],
						'order' => $settings['order'],
						'per_page' => $settings['limit'],
						'source' => $settings['source'],
						'temp' => 'portfolio',
						'img_size' => '',
						'cat_ids' => $cat_ids
					);

					wp_enqueue_script( 'arcdeco-portfolio-infinite-scroll-el', get_template_directory_uri() . '/assets/js/portfolio-infinite-scroll-el.js', array( 'jquery' ), '1.0', true );
					wp_localize_script( 'arcdeco-portfolio-infinite-scroll-el', 'ajax_portfolio_infinite_scroll_data', $infinite_scrolling_data );
				endif;
			else :
		    	get_template_part( 'template-parts/content', 'none' );
			endif; wp_reset_postdata(); ?>
		</div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Portfolio_Widget() );