<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Arcdeco Service Widget.
 *
 * @since 1.0
 */

class Arcdeco_Services_List_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-services-list';
	}

	public function get_title() {
		return esc_html__( 'Services List', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fas fa-list';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your title', 'arcdeco-plugin' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'num', [
				'label'       => esc_html__( 'Num', 'arcdeco-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter num', 'arcdeco-plugin' ),
				'default' => esc_html__( 'Enter num', 'arcdeco-plugin' ),
			]
		);

		$repeater->add_control(
			'text', [
				'label'       => esc_html__( 'Text', 'arcdeco-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter text', 'arcdeco-plugin' ),
				'default' => esc_html__( 'Enter text', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'arcdeco-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ text }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label' => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .section-about .title--h1' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .section-about .title--h1',
			]
		);			

		$this->end_controls_section();

		$this->start_controls_section(
			'list_styling',
			[
				'label' => esc_html__( 'Num', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'list_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .list-row .list-row__col-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'list_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .list-row .list-row__col-title',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() { 
		$settings = $this->get_settings_for_display();
		
		$this->add_inline_editing_attributes( 'title', 'basic' );

		?>

		<!-- Services List -->
		<div class="container section-about">
		    <div class="row">
			    <div class="col-12 col-lg-8 col-xl-6">
			    	<?php if ( $settings['title'] ) : ?>
				    <h1 class="title title--h1 js-lines">
				    	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>><?php echo wp_kses_post( $settings['title'] ); ?></span>
				    </h1>
					<?php endif; ?>
				</div>
			    <div class="col-12 col-xl-6">
			    	<?php if ( $settings['items'] ) : ?>
				    <ul class="list-row">
					    <?php 
					        foreach ( $settings['items'] as $index => $item ) : 
					        $item_num = $this->get_repeater_setting_key( 'num', 'items', $index );
					    	$this->add_inline_editing_attributes( $item_num, 'none' );

					    	$item_text = $this->get_repeater_setting_key( 'text', 'items', $index );
					    	$this->add_inline_editing_attributes( $item_text, 'basic' );
					    ?>
					    <li class="list-row__item js-list">
						    <?php if ( $item['num'] ) : ?>
						    <span class="list-row__col-num weight--600">
						    	<span <?php echo $this->get_render_attribute_string( $item_num ); ?>>
			    					<?php echo wp_kses_post( $item['num'] ); ?>
			    				</span>
						    </span>
							<?php endif; ?>
							<?php if ( $item['text'] ) : ?>
							<h4 class="list-row__col-title title title--h5 weight--600">
								<span <?php echo $this->get_render_attribute_string( $item_text ); ?>>
			    					<?php echo wp_kses_post( $item['text'] ); ?>
			    				</span>
							</h4>
							<?php endif; ?>
						</li>
						<?php endforeach; ?>
					</ul>
					<?php endif; ?>
				</div>
			</div>
		</div>
		
		<?php 
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() { ?>

		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>

		<!-- Services List -->
		<div class="container section-about">
		    <div class="row">
			    <div class="col-12 col-lg-8 col-xl-6">
			    	<# if( settings.title ) { #>
				    <h1 class="title title--h1 js-lines">
				    	<span {{{ view.getRenderAttributeString( 'title' ) }}}>{{{ settings.title }}}</span>
				    </h1>
					<# } #>
				</div>
			    <div class="col-12 col-xl-6">
			    	<# if ( settings.items ) { #>
				    <ul class="list-row">
						<# _.each( settings.items, function( item, index ) {
					        var item_num = view.getRepeaterSettingKey( 'num', 'items', index );
							view.addInlineEditingAttributes( item_num, 'none' );

							var item_text = view.getRepeaterSettingKey( 'text', 'items', index );
							view.addInlineEditingAttributes( item_text, 'basic' );
						#>
					    <li class="list-row__item js-list">
						    <# if ( item.num ) { #>
						    <span class="list-row__col-num weight--600">
						    	<span {{{ view.getRenderAttributeString( item_num ) }}}>{{{ item.num }}}</span>
						    </span>
							<# } #>
							<# if ( item.text ) { #>
							<h4 class="list-row__col-title title title--h5 weight--600">
								<span {{{ view.getRenderAttributeString( item_text ) }}}>{{{ item.text }}}</span>
							</h4>
							<# } #>
						</li>
						<# }); #>
					</ul>
					<# } #>
				</div>
			</div>
		</div>

	<?php }
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Services_List_Widget() );