<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Arcdeco About Heading Widget.
 *
 * @since 1.0
 */
class Arcdeco_Started_Section_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-started-section';
	}

	public function get_title() {
		return esc_html__( 'Started Section', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fab fa-stripe-s';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Title', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h1',
				'options' => [
					'h1'  => __( 'H1', 'arcdeco-plugin' ),
					'h2' => __( 'H2', 'arcdeco-plugin' ),
					'h3' => __( 'H3', 'arcdeco-plugin' ),
					'div' => __( 'DIV', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter your description', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Type your description here', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'arcdeco-plugin' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter subtitle', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Subtitle', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'image',
			[
				'label'       => esc_html__( 'Image', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_styling',
			[
				'label'     => esc_html__( 'Started Section', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label'   => esc_html__( 'Alignment', 'arcdeco-plugin' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'arcdeco-plugin' ),
						'icon'  => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'arcdeco-plugin' ),
						'icon'  => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'arcdeco-plugin' ),
						'icon'  => 'fa fa-align-right',
					],
				],
				'default' => 'left',
				'prefix_class' => 'elementor-align%s-',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .header-page .title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .header-page .title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'description_styling',
			[
				'label'     => esc_html__( 'Description', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'description_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .header-page .foreword' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'description_typography',
				'selector' => '{{WRAPPER}} .header-page .foreword',
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'subtitle_styling',
			[
				'label'     => esc_html__( 'Subtitle', 'arcdeco-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'subtitle_color',
			[
				'label'     => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .header-page .text-rotate' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'subtitle_typography',
				'selector' => '{{WRAPPER}} .header-page .text-rotate',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		$this->add_inline_editing_attributes( 'subtitle', 'none' );
		$this->add_inline_editing_attributes( 'description', 'advanced' );

		?>

		<header class="header-page">
		    <div class="container">
			    <div class="row">
			    	<?php if ( $settings['title'] ) : ?>
				    <div class="col-12">
	                    <<?php echo esc_attr( $settings['title_tag'] ); ?> class="title title--display-1 js-lines">
	                    	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
	                    		<?php echo wp_kses_post( $settings['title'] ); ?>
	                    	</span>
	                    </<?php echo esc_attr( $settings['title_tag'] ); ?>>
					</div>
					<?php endif; ?>
					<?php if ( $settings['description'] ) : ?>
					<div class="col-12">
					    <div class="foreword js-lines">
					    	<div <?php echo $this->get_render_attribute_string( 'description' ); ?>>
					    		<?php echo wp_kses_post( $settings['description'] ); ?>		
					    	</div>
					    </div>
					</div>
					<?php endif; ?>
                </div>
            </div>				
			<div class="image-gutter">
				<?php if ( $settings['image'] ) : 
					$image = wp_get_attachment_image_url( $settings['image']['id'], 'arcdeco_1920xAuto' );
				?>
			    <div class="reveal-box">
			        <img class="cover" src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $settings['subtitle'] ); ?>" />
				    <span class="reveal-overlay"></span>
				</div>
				<?php endif; ?>
				<?php if ( $settings['subtitle'] ) : ?>
				<span class="text-rotate">
					<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>>
						<?php echo esc_html( $settings['subtitle'] ); ?>
					</span>
				</span>
				<?php endif; ?>
			</div>
		</header>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() {
		?>
		
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		view.addInlineEditingAttributes( 'subtitle', 'none' );
		view.addInlineEditingAttributes( 'description', 'advanced' );
		#>

		<header class="header-page">
		    <div class="container">
			    <div class="row">
			    	<# if ( settings.title ) { #>
				    <div class="col-12">
	                    <{{{ settings.title_tag }}} class="title title--display-1 js-lines">
	                    	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
	                    		{{{ settings.title }}}
	                    	</span>
	                    </{{{ settings.title_tag }}}>
					</div>
					<# } #>
					<# if ( settings.description ) { #>
					<div class="col-12">
					    <div class="foreword js-lines">
					    	<div {{{ view.getRenderAttributeString( 'description' ) }}}>
					    		{{{ settings.description }}}
					    	</div>
					    </div>
					</div>
					<# } #>
                </div>
            </div>				
			<div class="image-gutter">
			    <# if ( settings.image ) { #>
			    <div class="reveal-box">
			        <img class="cover" src="{{{ settings.image.url }}}" alt="{{{ settings.subtitle }}}" />
				    <span class="reveal-overlay"></span>
				</div>
				<# } #>
				<# if ( settings.subtitle ) { #>
				<span class="text-rotate">
					<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>
						{{{ settings.subtitle }}}
					</span>
				</span>
				<# } #>
			</div>
		</header>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Started_Section_Widget() );