<?php
/**
 * arcdeco functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package arcdeco
 */

if ( ! function_exists( 'arcdeco_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function arcdeco_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on arcdeco, use a find and replace
		 * to change 'arcdeco' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'arcdeco', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' => esc_html__( 'Primary Menu', 'arcdeco' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Image Sizes
		add_image_size( 'arcdeco_800x800', 800, 800, true );
		add_image_size( 'arcdeco_900xAuto', 900, 9999, false );
		add_image_size( 'arcdeco_1170x658', 1170, 658, true );
		add_image_size( 'arcdeco_1920xAuto', 1920, 9999, false );
	}
endif;
add_action( 'after_setup_theme', 'arcdeco_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function arcdeco_content_width() {
	// This variable is intended to be overruled from themes.
	// Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'arcdeco_content_width', 900 );
}
add_action( 'after_setup_theme', 'arcdeco_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function arcdeco_widgets_init() {
	register_sidebar( array(
		'name'		  => esc_html__( 'Sidebar', 'arcdeco' ),
		'id'			=> 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'arcdeco' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	if ( class_exists( 'WooCommerce' ) ) :
		register_sidebar( array(
			'name'          => esc_html__( 'Shop Sidebar', 'arcdeco' ),
			'id'            => 'shop-sidebar',
			'description'   => esc_html__( 'Sidebar that appears on the shop.', 'arcdeco' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h1 class="widget-title">',
			'after_title'   => '</h1>',
		) );
	endif;
}
add_action( 'widgets_init', 'arcdeco_widgets_init' );

/**
 * Register Default Fonts
 */
function arcdeco_fonts_url() {
	$fonts_url = '';

	/* Translators: If there are characters in your language that are not
	 * supported by Lora, translate this to 'off'. Do not translate
	 * into your own language.
	 */
	$roboto = _x( 'on', 'Roboto: on or off', 'arcdeco' );

	if ( 'off' !== $roboto ) {
		$font_families = array();

		$font_families[] = 'Roboto:300,400,500,700,900';

		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( 'latin,latin-ext' ),
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}

/**
 * Enqueue scripts and styles.
 */
function arcdeco_stylesheets() {
	// Web fonts
	wp_enqueue_style( 'arcdeco-fonts', arcdeco_fonts_url(), array(), null );

	$headingsFont =  get_field( 'primary_font_family', 'options' );
	$paragraphsFont =  get_field( 'text_font_family', 'options' );

	// Custom fonts
	if ( $headingsFont ) {
		wp_enqueue_style( 'arcdeco-heading-font', $headingsFont['url'] , array(), null );
	}
	if ( $paragraphsFont ) {
		wp_enqueue_style( 'arcdeco-paragraph-font', $paragraphsFont['url'] , array(), null );
	}

	/*Styles*/
	wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/assets/styles/bootstrap.css', '1.0' );
	wp_enqueue_style( 'swiper', get_template_directory_uri() . '/assets/styles/swiper.css', '1.0' );
	wp_enqueue_style( 'arcdeco-fonticons', get_template_directory_uri() . '/assets/icons/fonticons.css', '1.0' );
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/assets/fonts/font-awesome/css/font-awesome.css', '1.0' );
	wp_enqueue_style( 'arcdeco-style', get_stylesheet_uri(), array( 'bootstrap', 'swiper', 'arcdeco-fonticons', 'font-awesome' ) );
	wp_enqueue_style( 'formstyler', get_template_directory_uri() . '/assets/styles/jquery.formstyler.css', '1.0' );
}
add_action( 'wp_enqueue_scripts', 'arcdeco_stylesheets' );

function arcdeco_scripts() {
	$smooth_scroll =  get_field( 'smooth_scroll', 'option' );

	/*Default Scripts*/
	wp_enqueue_script( 'arcdeco-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), '20151215', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	/*Theme Scripts*/
	wp_enqueue_script( 'gsap', get_template_directory_uri() . '/assets/js/gsap.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'gsap-splitText', get_template_directory_uri() . '/assets/js/gsap.splitText.js', array('jquery'), '1.0.0', true );

	if ( $smooth_scroll ) {
		wp_enqueue_script( 'luxy-smooth-scroll', get_template_directory_uri() . '/assets/js/luxy-smooth-scroll.js', array('jquery'), '1.0.0', true );
		wp_enqueue_script( 'luxy-smooth-scroll-init', get_template_directory_uri() . '/assets/js/luxy-smooth-scroll-init.js', array('jquery'), '1.0.0', true );
	}

	wp_enqueue_script( 'lazysizes', get_template_directory_uri() . '/assets/js/lazysizes.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'autosize', get_template_directory_uri() . '/assets/js/autosize.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'swiper', get_template_directory_uri() . '/assets/js/swiper.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'images-loaded', get_template_directory_uri() . '/assets/js/imagesLoaded.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'jarallax', get_template_directory_uri() . '/assets/js/jarallax.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'jarallax-video', get_template_directory_uri() . '/assets/js/jarallax-video.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'object-fit-images', get_template_directory_uri() . '/assets/js/object-fit-images.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'medium-zoom', get_template_directory_uri() . '/assets/js/medium-zoom.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'scroll-magic', get_template_directory_uri() . '/assets/js/ScrollMagic.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'masonry-pkgd', get_template_directory_uri() . '/assets/js/masonry.pkgd.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'formstyler', get_template_directory_uri() . '/assets/js/jquery.formstyler.min.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'arcdeco-common', get_template_directory_uri() . '/assets/js/common.js', array('jquery'), '1.0.0', true );

	if ( is_singular() ) {
		$arcdeco_rrssb_init = 'jQuery(document).ready(function ($) { $(".social-share").rrssb({ title: "' . esc_attr( get_the_title() ) . '", url: "' . esc_url( get_the_permalink() ) . '" }); });';

		wp_enqueue_script( 'arcdeco-rrssb', get_template_directory_uri() . '/assets/js/rrssb.js', array('jquery'), '1.0.0', true );
		wp_add_inline_script('arcdeco-rrssb', $arcdeco_rrssb_init );
	}
}
add_action( 'wp_enqueue_scripts', 'arcdeco_scripts' );

/**
 * TGM
 */
require get_template_directory() . '/inc/plugins/plugins.php';

/**
 * ACF Options
 */

function arcdeco_acf_fa_version( $version ) {
 $version = 5;
 return $version;
}
add_filter( 'ACFFA_override_major_version', 'arcdeco_acf_fa_version' );

function arcdeco_acf_json_load_point( $paths ) {
	$paths = array( get_template_directory() . '/inc/acf-json' );
	if( is_child_theme() ) {
		$paths[] = get_stylesheet_directory() . '/inc/acf-json';
	}

	return $paths;
}

add_filter('acf/settings/load_json', 'arcdeco_acf_json_load_point');

if ( function_exists( 'acf_add_options_page' ) ) {
	// Hide ACF field group menu item
	add_filter( 'acf/settings/show_admin', '__return_false' );

	// Add ACF Options Page
	acf_add_options_page( array(
		'page_title' 	=> esc_html__( 'Theme Options', 'arcdeco' ),
		'menu_title'	=> esc_html__( 'Theme Options', 'arcdeco' ),
		'menu_slug' 	=> 'theme-options',
		'capability'	=> 'edit_theme_options',
	) );
}

function arcdeco_acf_json_save_point( $path ) {
	// update path
	$path = get_stylesheet_directory() . '/inc/acf-json';

	// return
	return $path;
}
add_filter( 'acf/settings/save_json', 'arcdeco_acf_json_save_point' );

function arcdeco_acf_fallback() {
	// ACF Plugin fallback
	if ( ! is_admin() && ! function_exists( 'get_field' ) ) {
		function get_field( $field = '', $id = false ) {
			return false;
		}
		function the_field( $field = '', $id = false ) {
			return false;
		}
		function have_rows( $field = '', $id = false ) {
			return false;
		}
		function has_sub_field( $field = '', $id = false ) {
			return false;
		}
		function get_sub_field( $field = '', $id = false ) {
			return false;
		}
		function the_sub_field( $field = '', $id = false ) {
			return false;
		}
	}
}
add_action( 'init', 'arcdeco_acf_fallback' );

/**
 * OCDI
 */
require get_template_directory() . '/inc/ocdi-setup.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Include Skin Options
 */
require get_template_directory() . '/inc/skin-options.php';

/**
 * Include Infinite Scroll
 */
require get_template_directory() . '/inc/infinite-scroll.php';

/**
 * Get Archive Title
 */

function arcdeco_archive_title( $title ) {
	if ( is_category() ) {
		$title = single_cat_title( '', false );
	} elseif ( is_post_type_archive( 'portfolio' ) ) {
		$title = post_type_archive_title( '', false );
	} elseif ( is_tag() ) {
		$title = single_tag_title( esc_html__( 'Tag: ', 'arcdeco' ), false );
	} elseif ( is_author() ) {
		$title = esc_html__( 'Author: ', 'arcdeco' ) . get_the_author();
	}

	return $title;
}
add_filter( 'get_the_archive_title', 'arcdeco_archive_title' );

/**
 * Excerpt
 */
function arcdeco_custom_excerpt_length( $length ) {
	return 44;
}
add_filter( 'excerpt_length', 'arcdeco_custom_excerpt_length' );

function arcdeco_new_excerpt_more( $more ) {
	return esc_html__( '... ', 'arcdeco' ) . '<div><a href="' . esc_url( get_permalink() ) . '" class="btn-link">' . esc_html__( 'READ MORE', 'arcdeco' ) . '</a></div>';
}
add_filter( 'excerpt_more', 'arcdeco_new_excerpt_more' );

/**
 * Add class to sub-menu
 */
function arcdeco_change_wp_nav_menu( $classes, $args, $depth ) {
	if ( $args->theme_location == 'primary' && get_field( 'menu_type', 'option' ) == 3 ) {
		$classes[] = 'sideNav__sub closed';
	} else if( $args->theme_location == 'primary' && get_field( 'menu_type', 'option' ) == 2 ) {
		$classes[] = 'dropdown-menu';
	} else if( $args->theme_location == 'primary' && get_field( 'menu_type', 'option' ) == 1 ) {
		$classes[] = 'dropdown-menu';
	}
	return $classes;
}
add_filter( 'nav_menu_submenu_css_class', 'arcdeco_change_wp_nav_menu', 10, 3 );

/**
 * Top Menu
 */
class arcdeco_Topmenu_Walker extends Walker_Nav_menu {

	function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		global $wp_query;

		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';
		$class_names = '';
		if ( isset( $args->link_after ) ) {
			$args->link_after = '';
		}

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				array_push( $item->classes, 'dropdown' );
			}
		}

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;
		$classes[] = 'nav-overlay__menu-item';
		$class_names = join(' ', $classes);

	   	$class_names = ' class="'. esc_attr( $class_names ) . '"';
		$attributes = ! empty( $item->target ) ? ' target="' . esc_attr( $item->target ) . '"' : '';
		$attributes .= ! empty( $item->xfn ) ? ' rel="' . esc_attr( $item->xfn ) . '"' : '';
		$attributes .= ' data-text="' . esc_attr( apply_filters( 'the_title', $item->title, $item->ID ) ) . '"';

		$index = $item->menu_order;
		if ( $index < 10 ) {
			$index = '0' . $index;
		}

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				$args->link_after = ' <i class="icon icon-down-open"></i>';
			}
		}

		if ( has_nav_menu( 'primary' ) ) {
			$output .= $indent . '<li id="menu-item-'. esc_attr( $item->ID ) . '"' . $class_names . '>';

			$attributes .= ! empty( $item->url ) ? ' href="' . esc_url( $item->url ) . '"' : '';

			$item_output = $args->before;
			$item_output = '<span class="num">' . esc_html( $index ) . '</span>';
			$item_output .= '<a' . $attributes . '>';
			$item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID );
			$item_output .= $args->link_after;
			$item_output .= '</a>';
			$item_output .= $args->after;

			$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
		}
	}

}

/**
 * Top Menu Horizontal
 */
class arcdeco_TopmenuHorizontal_Walker extends Walker_Nav_menu {

	function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		global $wp_query;

		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';
		$class_names = '';
		if ( isset( $args->link_after ) ) {
			$args->link_after = '';
		}

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				array_push( $item->classes, 'dropdown' );
			}
		}

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;
		$classes[] = 'nav__item';
		$class_names = join(' ', $classes);

	   	$class_names = ' class="'. esc_attr( $class_names ) . '"';
		$attributes = ! empty( $item->target ) ? ' target="' . esc_attr( $item->target ) . '"' : '';
		$attributes .= ! empty( $item->xfn ) ? ' rel="' . esc_attr( $item->xfn ) . '"' : '';
		$attributes_classes = 'nav__link';

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				$attributes_classes .= ' dropdown-toggle';
				$args->link_after = ' <i class="icon icon-down-open"></i>';
			}
		}

		$attributes .= ' class="' . esc_attr( $attributes_classes ) . '"';

		if ( has_nav_menu( 'primary' ) ) {
			$output .= $indent . '<li id="menu-item-'. esc_attr( $item->ID ) . '"' . $class_names . '>';

			$attributes .= ! empty( $item->url ) ? ' href="' . esc_url( $item->url ) . '"' : '';

			$item_output = $args->before;
			$item_output .= '<a' . $attributes . '>';
			$item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID );
			$item_output .= $args->link_after;
			$item_output .= '</a>';
			$item_output .= $args->after;

			$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
		}
	}
}

/**
 * Top Menu Side
 */
class arcdeco_TopmenuSide_Walker extends Walker_Nav_menu {

	function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		global $wp_query;

		$indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';
		$class_names = '';
		if ( isset( $args->link_after ) ) {
			$args->link_after = '';
		}

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				array_push( $item->classes, 'dropdown' );
			}
		}

		$classes = empty( $item->classes ) ? array() : (array) $item->classes;
		$classes[] = 'sideNav__item';
		$class_names = join(' ', $classes);

	   	$class_names = ' class="'. esc_attr( $class_names ) . '"';
		$attributes = ! empty( $item->target ) ? ' target="' . esc_attr( $item->target ) . '"' : '';
		$attributes .= ! empty( $item->xfn ) ? ' rel="' . esc_attr( $item->xfn ) . '"' : '';
		$attributes_classes = '';

		if ( is_array($item->classes) ) {
			if ( in_array('menu-item-has-children', $item->classes ) ) {
				$attributes_classes .= ' sideNav-collapsed';
				$args->link_after = ' <i class="icon icon-down-open"></i>';
			}
		}

		$attributes .= ' class="' . esc_attr( $attributes_classes ) . '"';

		if ( has_nav_menu( 'primary' ) ) {
			$output .= $indent . '<li id="menu-item-'. esc_attr( $item->ID ) . '"' . $class_names . '>';

			if ( $attributes_classes == '' ) {
				$attributes .= ! empty( $item->url ) ? ' href="' . esc_url( $item->url ) . '"' : '';
			}

			$item_output = $args->before;
			$item_output .= '<a' . $attributes . '>';
			$item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID );
			$item_output .= $args->link_after;
			$item_output .= '</a>';
			$item_output .= $args->after;

			$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
		}
	}
}

/**
 * Woocommerce Support
 */

function arcdeco_add_woocommerce_support() {
	add_theme_support( 'woocommerce', array(
		'thumbnail_image_width' => 800,
		'gallery_thumbnail_image_width' => 200,
		'single_image_width' => 800,
		'product_grid' => array(
			'default_rows' => 2,
			'min_rows' => 2,
			'max_rows' => 10,
			'default_columns' => 2,
			'min_columns' => 2,
			'max_columns' => 6,
		),
	) );
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'arcdeco_add_woocommerce_support' );

/**
 * Update contents AJAX mini-cart
 */

function arcdeco_woocommerce_update_count_mini_cart( $fragments ) {
	ob_start();
	?>

	<span class="cart-count">
		<?php echo sprintf (_n( '%d', '%d', WC()->cart->get_cart_contents_count(), 'arcdeco' ), WC()->cart->get_cart_contents_count() ); ?>
	</span>

	<?php
	$fragments['span.cart-count'] = ob_get_clean();
	return $fragments;
}
add_filter( 'woocommerce_add_to_cart_fragments', 'arcdeco_woocommerce_update_count_mini_cart' );

function arcdeco_woocommerce_update_content_mini_cart( $fragments ) {
	ob_start();
	?>

	<div class="cart-widget">
       <?php woocommerce_mini_cart(); ?>
    </div>

	<?php
	$fragments['div.cart-widget'] = ob_get_clean();
	return $fragments;
}
add_filter( 'woocommerce_add_to_cart_fragments', 'arcdeco_woocommerce_update_content_mini_cart' );

/* Custom Body Class  */
function arcdeco_body_class_names( $classes ) {
	$animation_disabled = 0;
	if ( get_field( 'animation_disabled', 'option' ) ) {
	  $animation_disabled = 'animation--disabled';
	}
	if ( $animation_disabled ) {
		$classes[] = $animation_disabled;
	}

	return $classes;
}
add_filter( 'body_class', 'arcdeco_body_class_names' );
