<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Arcdeco Hero Section Widgets.
 *
 * @since 1.0
 */
class Arcdeco_Hero_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-hero';
	}

	public function get_title() {
		return esc_html__( 'Hero Section', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fa fa-tv';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your title', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Architecture and decor.', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label'       => esc_html__( 'Subtitle', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your Subtitle', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'Introduction', 'arcdeco-plugin' ),
			]
		);
		$this->add_control(
			'description',
			[
				'label'       => esc_html__( 'Description', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your Description', 'arcdeco-plugin' ),
				'default'     => esc_html__( 'The essence of interior design will always be about people and how they live.', 'arcdeco-plugin' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'video_tab',
			[
				'label' => esc_html__( 'Background', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);	

		$this->add_control(
			'bg',
			[
				'label' => esc_html__( 'Background', 'arcdeco-plugin' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'image',
				'options' => [
					'image'  => esc_html__( 'Image', 'arcdeco-plugin' ),
					'video' => esc_html__( 'Video', 'arcdeco-plugin' ),
				],
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose Image', 'arcdeco-plugin' ),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
				'condition' => ['bg' => 'image']
			]
		);

		$this->add_control(
			'video',
			[
				'label' => esc_html__( 'Video Link', 'arcdeco-plugin' ),
				'type' => \Elementor\Controls_Manager::URL,
				'show_external' => false,
				'condition' => ['bg' => 'video']
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'social_tab',
			[
				'label' => esc_html__( 'Social', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);	

		$this->add_control(
			'social',
			[
				'label' => esc_html__( 'Show Social', 'arcdeco-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'arcdeco-plugin' ),
				'label_off' => esc_html__( 'Hide', 'arcdeco-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label' => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .hero-intro .hero__header .title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .hero-intro .hero__header .title',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'subtitle_styling',
			[
				'label' => esc_html__( 'Subtitle', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'subtitle_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .hero-intro .note .title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'subtitle_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .hero-intro .note .title',
			]
		);			

		$this->end_controls_section();

		$this->start_controls_section(
			'description_styling',
			[
				'label' => esc_html__( 'Description', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-intro .note .description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'description_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .hero-intro .note' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .hero-intro .note:before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .hero-intro .note .description',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() { 
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'none' );
		$this->add_inline_editing_attributes( 'subtitle', 'none' );
		$this->add_inline_editing_attributes( 'description', 'basic' );

		$social_links = get_field( 'social_links', 'option' );

		?>
		<!-- Intro -->
        <header class="hero hero-intro overlay--45">
		    <?php if( $settings['title'] ) : ?>
		    <div class="hero__header">
			    <h1 class="title title--display-1 js-lines<?php if( ! $settings['subtitle'] && ! $settings['description'] ) : ?> mb-4<?php endif; ?>">
			    	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>><?php echo wp_kses_post( $settings['title'] ); ?></span>
			    </h1>
			</div>
			<?php endif; ?>
			
			<?php if( $settings['subtitle'] || $settings['description'] ) : ?>
			<div class="note">
			    <div class="note__item js-scroll-show">
			    	<?php if( $settings['subtitle'] ) : ?>
                    <h4 class="title title--display-1">
                    	<span <?php echo $this->get_render_attribute_string( 'subtitle' ); ?>><?php echo wp_kses_post( $settings['subtitle'] ); ?></span>
                    </h4>
                	<?php endif; ?>
                	<?php if( $settings['description'] ) : ?>
                    <p class="description">
                    	<span <?php echo $this->get_render_attribute_string( 'description' ); ?>><?php echo wp_kses_post( $settings['description'] ); ?></span>
                    </p>
                    <?php endif; ?>
                </div>
			</div>
			<?php endif; ?>
			
			<?php if ( $settings['social'] == 'yes' && $social_links ) : ?>
			<!-- Social -->
			<div class="social social-floating">
				<?php foreach ( $social_links as $social_link ) : ?>
		        <a class="social__link" target="blank" href="<?php echo esc_url( $social_link['url'] ); ?>"><i class="<?php echo esc_attr( $social_link['icon'] ); ?>"></i></a>
		    	<?php endforeach; ?>
		    </div>
		    <?php endif; ?>
			
			<?php if( $settings['bg'] == 'video' ) : ?>
			<!-- Video -->
			<video id="js-video-bg" class="video-background js-parallax" playsinline autoplay loop muted>
                <source src="<?php echo esc_url( $settings['video']['url'] ); ?>" type="video/mp4">
            </video>
            <?php endif; ?>
            <?php if( $settings['bg'] == 'image' && $settings['image'] ) :
            	$image = wp_get_attachment_image_url( $settings['image']['id'], 'arcdeco_1920xAuto' );
            ?>
            <!-- Main image -->
            <img class="hero__image cover js-scale js-parallax" src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $settings['title'] ); ?>" />
            <?php endif; ?>
		</header>
		<?php 
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() { ?>
		
		<#
		view.addInlineEditingAttributes( 'title', 'none' );
		view.addInlineEditingAttributes( 'subtitle', 'none' );
		view.addInlineEditingAttributes( 'description', 'basic' );
		#>

		<!-- Intro -->
        <header class="hero hero-intro overlay--45">
        	<# if( settings.title ) { #>
		    <div class="hero__header">
			    <h1 class="title title--display-1 js-lines">
			    	<span {{{ view.getRenderAttributeString( 'title' ) }}}>{{{ settings.title }}}</span>
			    </h1>
			</div>
			<# } #>
			
			<# if( settings.subtitle || settings.description ) { #>
			<div class="note">
			    <div class="note__item js-scroll-show">
			    	<# if( settings.subtitle ) { #>
                    <h4 class="title title--display-1">
                    	<span {{{ view.getRenderAttributeString( 'subtitle' ) }}}>{{{ settings.subtitle }}}</span>
                    </h4>
                	<# } #>
                	<# if( settings.description ) { #>
                    <p class="description">
                    	<span {{{ view.getRenderAttributeString( 'description' ) }}}>
                    		{{{ settings.description }}}
                    	</span>
                    </p>
                    <# } #>
                </div>
			</div>
			<# } #>
			
			<# if( settings.social.length ) { #>
			<!-- Social -->
			<div class="social social-floating">
				<# _.each( settings.social, function( item ) { #>
		        <a class="social__link" href="{{{ item.link }}}"><i class="{{{ item.icon }}}"></i></a>
		    	<# }); #>
		    </div>
		    <# } #>
			
			<# if( settings.bg == 'video' ) { #>
			<!-- Video -->
			<video id="js-video-bg" class="video-background js-parallax" playsinline autoplay loop muted>
                <source src="{{{ settings.video.url }}}" type="video/mp4">
            </video>
            <# } #>
            <# if( settings.bg == 'image' ) { #>
            <!-- Main image -->
            <img class="hero__image cover js-scale js-parallax" src="{{{ settings.image.url }}}" alt="{{{ settings.title }}}" />
            <# } #>
		</header>
	<?php }
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Hero_Widget() );