<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Arcdeco Service Widget.
 *
 * @since 1.0
 */

class Arcdeco_Services_Grid_Widget extends Widget_Base {

	public function get_name() {
		return 'arcdeco-services-grid';
	}

	public function get_title() {
		return esc_html__( 'Services Grid', 'arcdeco-plugin' );
	}

	public function get_icon() {
		return 'fas fa-th-large';
	}

	public function get_categories() {
		return [ 'arcdeco-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_tab',
			[
				'label' => esc_html__( 'Content', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label'       => esc_html__( 'Image', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'arcdeco-plugin' ),
				'default'	=> esc_html__( 'Enter title', 'arcdeco-plugin' ),
			]
		);

		$repeater->add_control(
			'description', [
				'label'       => esc_html__( 'Description', 'arcdeco-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter description', 'arcdeco-plugin' ),
				'default'	=> esc_html__( 'Enter description', 'arcdeco-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Items', 'arcdeco-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label' => esc_html__( 'Title', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .service-item .title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .service-item .title',
			]
		);			

		$this->end_controls_section();

		$this->start_controls_section(
			'description_styling',
			[
				'label' => esc_html__( 'Description', 'arcdeco-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);		

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__( 'Color', 'arcdeco-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .service-item .description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'label' => esc_html__( 'Typography:', 'arcdeco-plugin' ),
				'selector' => '{{WRAPPER}} .service-item .description',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		
		?>
		
		<!-- Services Grid -->
		<div class="container section-about">
			<?php if ( $settings['items'] ) : ?>
		    <div class="row justify-content-between">
			    <?php 
			        foreach ( $settings['items'] as $index => $item ) : 
			        $item_title = $this->get_repeater_setting_key( 'title', 'items', $index );
			    	$this->add_inline_editing_attributes( $item_title, 'basic' );

			    	$item_description = $this->get_repeater_setting_key( 'description', 'items', $index );
			    	$this->add_inline_editing_attributes( $item_description, 'basic' );
			    ?>
			    <div class="col-12 col-md-5 service-item js-scroll-show">
				    <?php if ( $item['image'] ) : ?>
				    <img class="service-item__icon" src="<?php echo esc_url( $item['image']['url'] ); ?>" alt="<?php echo esc_attr__( 'Image', 'arcdeco-plugin' ); ?>" />
				    <?php endif; ?>
				    <?php if ( $item['title'] ) : ?>
			        <h3 class="title title--h4 js-lines">
			        	<span <?php echo $this->get_render_attribute_string( $item_title ); ?>>
			    			<?php echo wp_kses_post( $item['title'] ); ?>
			    		</span>
			        </h3>
			        <?php endif; ?>
			        <?php if ( $item['description'] ) : ?>
					<p class="description js-lines">
						<span <?php echo $this->get_render_attribute_string( $item_description ); ?>>
			    			<?php echo wp_kses_post( $item['description'] ); ?>
			    		</span>
					</p>
					<?php endif; ?>
			    </div>
				<?php endforeach; ?>			
			</div>
			<?php endif; ?>	
		</div>

		<?php 
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _content_template() { ?>

		<!-- Services Grid -->
		<div class="container section-about">
			<# if( settings.items ) { #>
		    <div class="row justify-content-between">
			    <# _.each( settings.items, function( item, index ) {
			        var item_title = view.getRepeaterSettingKey( 'title', 'items', index );
					view.addInlineEditingAttributes( item_title, 'basic' );

					var item_description = view.getRepeaterSettingKey( 'description', 'items', index );
					view.addInlineEditingAttributes( item_description, 'basic' );
				#>
			    <div class="col-12 col-md-5 service-item js-scroll-show">
				    <# if( item.image ) { #>
				    <img class="service-item__icon" src="{{{ item.image.url }}}" alt="" />
				    <# } #>
				    <# if( item.title ) { #>
			        <h3 class="title title--h4 js-lines">
			        	<span {{{ view.getRenderAttributeString( item_title ) }}}>{{{ item.title }}}</span>
			        </h3>
			        <# } #>
			        <# if( item.description ) { #>
					<p class="description js-lines">
						<span {{{ view.getRenderAttributeString( item_description ) }}}>{{{ item.description }}}</span>
					</p>
					<# } #>
			    </div>
				<# }); #>
			</div>
			<# } #>
		</div>

	<?php }
}

Plugin::instance()->widgets_manager->register_widget_type( new Arcdeco_Services_Grid_Widget() );