<?php
/**
 * Hellix Customizer Class
 *
 * @package  hellix
 * @since    1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Hellix_Customizer' ) ) :

	/**
	 * The hellix Customizer class
	 */
	class Hellix_Customizer {

		/**
		 * Setup class.
		 *
		 * @since 1.0
		 */
		public function __construct() {
			$this->init_hooks();
		}

		/**
		 * Init Hellix_Customizer when WordPress Initializes
		 */
		public function init_hooks() {
			add_action( 'customize_register', array( $this, 'customize_logos' ), 10 );
			add_action( 'customize_register', array( $this, 'customize_header' ), 10 );
			add_action( 'customize_register', array( $this, 'customize_general' ), 15 );
			add_action( 'customize_register', array( $this, 'customize_page_header' ), 20 );
			add_action( 'customize_register', array( $this, 'customize_blog' ), 30 );
			add_action( 'customize_register', array( $this, 'customize_portfolio' ), 40 );
			add_action( 'customize_register', array( $this, 'customize_footer' ), 50 );
			add_action( 'customize_register', array( $this, 'customize_404' ), 10 );
		}

				/**
				 * Customize site header
				 *
				 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
				 */
		public function customize_general( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_general',
				[
					'title'       => esc_html__( 'General', 'hellix' ),
					'description' => esc_html__( 'Customize general options.', 'hellix' ),
					'priority'    => 20,
				]
			);

			$this->add_general_section( $wp_customize );

		}

		/**
		 * Customizer Controls For Header.
		 *
		 *  @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_general_section( $wp_customize ) {

			$wp_customize->add_setting(
				'enable_hellix_page_loader',
				array(
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				)
			);

			$wp_customize->add_control(
				'enable_hellix_page_loader',
				array(
					'type'    => 'radio',
					'section' => 'hellix_general',
					'label'   => esc_html__( 'Page Loader', 'hellix' ),
					'choices' => array(
						'yes' => esc_html__( 'Enable', 'hellix' ),
						'no'  => esc_html__( 'Disable', 'hellix' ),
					),
				)
			);

			$wp_customize->selective_refresh->add_partial(
				'enable_hellix_page_loader',
				[
					'fallback_refresh' => true,
				]
			);
		}

		/**
		 * Customize all available site logos
		 *
		 * All logos located in title_tagline (Site Identity) section.
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_logos( $wp_customize ) {
			$this->add_customize_logos( $wp_customize );
		}

		/**
		 * Add Customize logos.
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_customize_logos( $wp_customize ) {

			// <editor-fold desc="mobile_logo">
			$wp_customize->add_setting(
				'mobile_logo',
				[
					'transport'         => 'postMessage',
					'theme_supports'    => 'custom-logo',
					'sanitize_callback' => 'absint',
				]
			);
			$wp_customize->add_control(
				new WP_Customize_Cropped_Image_Control(
					$wp_customize,
					'mobile_logo',
					[
						'section'       => 'title_tagline',
						'label'         => esc_html__( 'Mobile Logo', 'hellix' ),
						'description'   => esc_html__( 'Mobile logo inherits the same behavior for retina displays as desktop logo.', 'hellix' ),
						'priority'      => 9,
						'width'         => 24,
						'height'        => 24,
						'button_labels' => [
							'select'       => esc_html__( 'Select logo', 'hellix' ),
							'change'       => esc_html__( 'Change logo', 'hellix' ),
							'remove'       => esc_html__( 'Remove', 'hellix' ),
							'default'      => esc_html__( 'Default', 'hellix' ),
							'placeholder'  => esc_html__( 'No logo selected', 'hellix' ),
							'frame_title'  => esc_html__( 'Select logo', 'hellix' ),
							'frame_button' => esc_html__( 'Choose logo', 'hellix' ),
						],
					]
				)
			);
			$wp_customize->selective_refresh->add_partial(
				'mobile_logo',
				[
					'selector'            => '.mobile-logo-link',
					'container_inclusive' => true,
					'render_callback'     => 'hellix_mobile_logo',
				]
			);

		}

		/**
		 * Customize site footer
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_page_header( $wp_customize ) {

			$wp_customize->add_section(
				'hellix_header',
				[
					'title'       => esc_html__( 'Header', 'hellix' ),
					'description' => esc_html__( 'Customize the theme header.', 'hellix' ),
					'priority'    => 80,
				]
			);

			$this->add_header_section( $wp_customize );
		}

		/**
		 * Customize site footer
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_header( $wp_customize ) {

			$wp_customize->add_section(
				'hellix_header',
				[
					'title'       => esc_html__( 'Header', 'hellix' ),
					'description' => esc_html__( 'Customize the theme header.', 'hellix' ),
					'priority'    => 80,
				]
			);

			$this->add_header_section( $wp_customize );
		}

		/**
		 * Customize Controls for Header.
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_header_section( $wp_customize ) {
			/**
			 * Primary Color
			 */
			$header_image = get_header_image();
			$wp_customize->add_setting(
				'masthead_bg_color',
				array(
					'default'           => apply_filters( 'hellix_default_masthead_color', '#18130C' ),
					'sanitize_callback' => 'sanitize_hex_color',
				)
			);

			$wp_customize->add_control(
				new WP_Customize_Color_Control(
					$wp_customize,
					'masthead_bg_color',
					array(
						'label'    => esc_html__( 'Masthead Background color', 'hellix' ),
						'section'  => 'header_image',
						'settings' => 'masthead_bg_color',
					)
				)
			);

			$wp_customize->add_setting(
				'logo_variant',
				[
					'default'           => 'site_logo',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'logo_variant',
				[
					'type'        => 'select',
					'section'     => 'hellix_header',
					'label'       => esc_html__( 'Logo Variant', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to choose your Logo type.', 'hellix' ),
					'choices'     => [
						'site_logo'   => esc_html__( 'Site Logo', 'hellix' ),
						'site_icon'   => esc_html__( 'Site Icon', 'hellix' ),
						'hellix_icon' => esc_html__( 'Hellix Icon', 'hellix' ),
					],
				]
			);

			$wp_customize->add_setting(
				'navbar_variant',
				[
					'default'           => 'v1',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'navbar_variant',
				[
					'type'        => 'select',
					'section'     => 'hellix_header',
					'label'       => esc_html__( 'Header Variant', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to choose your header type.', 'hellix' ),
					'choices'     => [
						'v1'  => esc_html__( 'Header v1', 'hellix' ),
						'v2'  => esc_html__( 'Header v2', 'hellix' ),
						'v3'  => esc_html__( 'Header v3', 'hellix' ),
						'v4'  => esc_html__( 'Header v4', 'hellix' ),
						'v5'  => esc_html__( 'Header v5', 'hellix' ),
						'v6'  => esc_html__( 'Header v6', 'hellix' ),
						'v7'  => esc_html__( 'Header v7', 'hellix' ),
						'v8'  => esc_html__( 'Header v8', 'hellix' ),
						'v9'  => esc_html__( 'Header v9', 'hellix' ),
						'v10' => esc_html__( 'Header v10', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'navbar_variant',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'enable_topbar',
				[
					'default'           => 'yes',
					'sanitize_callback' => 'sanitize_key',
				]
			);
			$wp_customize->add_control(
				'enable_topbar',
				[
					'type'            => 'radio',
					'section'         => 'hellix_header',
					'label'           => esc_html__( 'Enable Topbar?', 'hellix' ),
					'choices'         => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'navbar_variant' ),
							[
								'v2',
								'v3',
							]
						);
					},
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'enable_topbar',
				[
					'selector'            => '.topbar',
					'container_inclusive' => true,
					'render_callback'     => 'hellix_topbar',
				]
			);

			$wp_customize->add_setting(
				'enable_search',
				[
					'default'           => 'yes',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'enable_search',
				[
					'type'    => 'radio',
					'section' => 'hellix_header',
					'label'   => esc_html__( 'Enable Search?', 'hellix' ),
					'choices' => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'enable_search',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'enable_offcanvas',
				[
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'enable_offcanvas',
				[
					'type'    => 'radio',
					'section' => 'hellix_header',
					'label'   => esc_html__( 'Enable Offcanvas?', 'hellix' ),
					'choices' => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'enable_offcanvas',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'hellix_header_sticky',
				[
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'hellix_header_sticky',
				[
					'type'    => 'radio',
					'section' => 'hellix_header',
					'label'   => esc_html__( 'Enable Sticky?', 'hellix' ),
					'choices' => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'hellix_header_sticky',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'enable_cart',
				[
					'default'           => 'yes',
					'sanitize_callback' => 'sanitize_key',
				]
			);
			$wp_customize->add_control(
				'enable_cart',
				[
					'type'        => 'radio',
					'section'     => 'hellix_header',
					'label'       => esc_html__( 'Show Shopping Cart?', 'hellix' ),
					'description' => esc_html__( 'Enable / disable cart in Header.', 'hellix' ),
					'choices'     => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);

			return in_array(
				get_theme_mod( 'navbar_variant' ),
				[
					'shop',
				]
			) && hellix_is_woocommerce_activated();

			$wp_customize->selective_refresh->add_partial(
				'enable_cart',
				[
					'fallback_refresh' => true,
				]
			);

		}

		/**
		 * Customize site blog
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_blog( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_blog',
				[
					/* translators: title of section in Customizer */
					'title'       => esc_html__( 'Blog', 'hellix' ),
					'description' => esc_html__( 'This section contains settings related to posts listing archives and single post.', 'hellix' ),
					'priority'    => 50,
				]
			);

			$this->add_blog_section( $wp_customize );
		}

		/**
		 * Adding Blog section controls
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_blog_section( $wp_customize ) {

			$wp_customize->add_setting(
				'blog_sidebar',
				[
					'default'           => 'right-sidebar',
					'sanitize_callback' => 'sanitize_key',
				]
			);
			$wp_customize->add_control(
				'blog_sidebar',
				[
					'type'            => 'select',
					'section'         => 'hellix_blog',
					/* translators: label field of control in Customizer */
					'label'           => esc_html__( 'Blog Sidebar', 'hellix' ),
					'description'     => esc_html__( 'This setting affects your blog page. This works when blog sidebar has widgets', 'hellix' ),
					'choices'         => [
						'left-sidebar'  => esc_html__( 'Left Sidebar', 'hellix' ),
						'right-sidebar' => esc_html__( 'Right Sidebar', 'hellix' ),
						'no-sidebar'    => esc_html__( 'No Sidebar', 'hellix' ),
					],
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'blog_layout' ),
							[
								'list',
							]
						);
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'blog_sidebar',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'blog_layout',
				[
					'default'           => 'list',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'blog_layout',
				[
					'type'        => 'select',
					'section'     => 'hellix_blog',
					/* translators: label field of control in Customizer */
					'label'       => esc_html__( 'Blog Layout', 'hellix' ),
					'description' => esc_html__( 'This setting affects both the posts listing (your blog page) and archives.', 'hellix' ),
					'choices'     => [
						/* translators: single item in a list of Blog Layout choices (in Customizer) */
						'grid' => esc_html__( 'Grid', 'hellix' ),
						/* translators: single item in a list of Blog Layout choices (in Customizer) */
						'list' => esc_html__( 'List', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'blog_layout',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_control(
				'hellix_single_post_social_share',
				[
					'type'        => 'checkbox',
					'section'     => 'hellix_blog',
					'label'       => esc_html__( 'Enable Social Share in Single Post', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to enable social share', 'hellix' ),

				]
			);

			$wp_customize->selective_refresh->add_partial(
				'hellix_single_post_social_share',
				[
					'fallback_refresh' => true,
				]
			);
		}

		/**
		 * Customize site header
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_portfolio( $wp_customize ) {
			$wp_customize->add_panel(
				'hellix_portfolio',
				array(
					'title'       => esc_html__( 'Portfolio', 'hellix' ),
					'description' => esc_html__( 'This section contains settings related to projects listing archives and single projects.', 'hellix' ),
					'priority'    => 40,

				)
			);
			$this->customize_single_portfolio( $wp_customize );
			$this->customize_style( $wp_customize );
			// $this->customize_hellix_layout( $wp_customize );
		}

		/**
		 * Customize Single portfolio
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_single_portfolio( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_single_portfolio',
				[
					/* translators: title of section in Customizer */
					'title'       => esc_html__( 'Single portfolio', 'hellix' ),
					'description' => esc_html__( 'This section contains settings related to posts listing archives and single post.', 'hellix' ),
					'priority'    => 50,
					'panel'       => 'hellix_portfolio',
				]
			);

			$this->add_single_portfolio_section( $wp_customize );
		}

		/**
		 * Customize Single Post
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_single_portfolio_section( $wp_customize ) {
			$this->static_contents = hellix_static_content_options();
			$wp_customize->add_setting(
				'single_portfolio_layout',
				[
					'default'           => 'v1',
					'sanitize_callback' => 'sanitize_key',
				]
			);
			$wp_customize->add_control(
				'single_portfolio_layout',
				[
					'type'    => 'select',
					'section' => 'hellix_single_portfolio',
					/* translators: label field of control in Customizer */
					'label'   => esc_html__( 'Portfolio Single Style', 'hellix' ),
					'choices' => [
						'v1' => esc_html__( 'Style v1', 'hellix' ),
						'v2' => esc_html__( 'Style v2', 'hellix' ),
						'v3' => esc_html__( 'Style v3', 'hellix' ),
						'v4' => esc_html__( 'Style v4', 'hellix' ),
						'v5' => esc_html__( 'Style v5', 'hellix' ),
						'v6' => esc_html__( 'Style v6', 'hellix' ),
						'v7' => esc_html__( 'Style v7', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'single_portfolio_layout',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'related_projects',
				[
					'default'           => 'yes',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'related_projects',
				[
					'type'        => 'radio',
					'section'     => 'hellix_single_portfolio',
					'label'       => esc_html__( 'Enable Related Projects', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to enable related projects', 'hellix' ),
					'choices'     => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'related_projects',
				[
					'fallback_refresh' => true,
				]
			);
		}
		/**
		 * Customize Single project
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_style( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_layout',
				[
					/* translators: title of section in Customizer */
					'title'       => esc_html__( 'Layout', 'hellix' ),
					'description' => esc_html__( 'This section contains settings related to single projects.', 'hellix' ),
					'priority'    => 60,
					'panel'       => 'hellix_portfolio',
				]
			);

			$this->add_portfolio_section( $wp_customize );
		}

		/**
		 * Adding Portfolio section controls
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_portfolio_section( $wp_customize ) {

			$wp_customize->add_setting(
				'portfolio_style',
				[
					'default'           => 'v1',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'portfolio_style',
				[
					'type'    => 'select',
					'section' => 'hellix_layout',
					/* translators: label field of control in Customizer */
					'label'   => esc_html__( 'Portfolio Style', 'hellix' ),
					'choices' => [
						'v1' => esc_html__( 'Style v1', 'hellix' ),
						'v2' => esc_html__( 'Style v2', 'hellix' ),
						'v3' => esc_html__( 'Style v3', 'hellix' ),
						'v4' => esc_html__( 'Style v4', 'hellix' ),
						'v5' => esc_html__( 'Style v5', 'hellix' ),
						'v6' => esc_html__( 'Style v6', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'portfolio_style',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'portfolio_container',
				[
					'default'           => 'yes',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'portfolio_container',
				[
					'type'            => 'radio',
					'section'         => 'hellix_layout',
					'label'           => esc_html__( 'Enable Container', 'hellix' ),
					'description'     => esc_html__( 'This setting allows you to enable container', 'hellix' ),
					'choices'         => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'portfolio_style' ),
							[
								'v1',
							]
						);
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'portfolio_container',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'portfolio_layout',
				[
					'default'           => '-col-3',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'portfolio_layout',
				[
					'type'            => 'select',
					'section'         => 'hellix_layout',
					/* translators: label field of control in Customizer */
					'label'           => esc_html__( 'Portfolio Layout', 'hellix' ),
					'choices'         => [
						'-col-2' => esc_html__( 'Two Columns', 'hellix' ),
						'-col-3' => esc_html__( 'Three Columns', 'hellix' ),
						'-col-4' => esc_html__( 'Four Columns', 'hellix' ),
					],
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'portfolio_style' ),
							[
								'v1',
								'v2',
								'v5',
								'v6',
							]
						);
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'portfolio_layout',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'portfolio_gap',
				[
					'default'           => '-gap-32',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'portfolio_gap',
				[
					'type'            => 'select',
					'section'         => 'hellix_layout',
					/* translators: label field of control in Customizer */
					'label'           => esc_html__( 'Portfolio Layout', 'hellix' ),
					'choices'         => [
						'-gap-16' => esc_html__( 'Gap 16', 'hellix' ),
						'-gap-32' => esc_html__( 'Gap 32', 'hellix' ),
						''        => esc_html__( 'No Gap', 'hellix' ),
					],
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'portfolio_style' ),
							[
								'v1',
								'v2',
								'v5',
								'v6',
							]
						);
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'portfolio_gap',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'portfolio_wrap_css',
				[
					'default'           => '-r-120',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'portfolio_wrap_css',
				[
					'type'            => 'text',
					'section'         => 'hellix_layout',
					'label'           => esc_html__( 'Wrap CSS', 'hellix' ),
					'active_callback' => function () {
						return in_array(
							get_theme_mod( 'portfolio_style' ),
							[
								'v1',
							]
						);
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'portfolio_wrap_css',
				[
					'fallback_refresh' => true,
				]
			);
		}

		/**
		 * Customize site 404
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_404( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_404',
				[
					'title'    => '404',
					'priority' => 31,
				]
			);

			$this->add_404_section( $wp_customize );
		}

		/**
		 * Adding 404 section controls.
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_404_section( $wp_customize ) {

			$wp_customize->selective_refresh->add_partial(
				'404_image_url',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'404_image_url',
				array(
					'transport'         => 'postMessage',
					'sanitize_callback' => 'absint',
				)
			);

			$wp_customize->add_control(
				new WP_Customize_Media_Control(
					$wp_customize,
					'404_image_url',
					array(
						'section'     => 'hellix_404',
						'label'       => esc_html__( '404 Image Upload', 'hellix' ),
						'description' => esc_html__(
							'This setting allows you to upload an image for 404 page.',
							'hellix'
						),
						'mime_type'   => 'image',
					)
				)
			);

			$wp_customize->add_setting(
				'404_title',
				[
					'default'           => esc_html_x( '404', 'front-end', 'hellix' ),
					'sanitize_callback' => 'sanitize_text_field',
					'transport'         => 'postMessage',
				]
			);
			$wp_customize->add_control(
				'404_title',
				[
					'type'    => 'text',
					'section' => 'hellix_404',
					'label'   => esc_html__( '404 Title', 'hellix' ),
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'404_title',
				[
					'render_callback' => function () {
						return esc_html( get_theme_mod( '404_title' ) );
					},
				]
			);

			$wp_customize->add_setting(
				'404_subtitle',
				[
					'default'           => esc_html_x( 'Oops! That page can’t be found.', 'front-end', 'hellix' ),
					'sanitize_callback' => 'sanitize_textarea_field',
					'transport'         => 'postMessage',
				]
			);
			$wp_customize->add_control(
				'404_subtitle',
				[
					'type'    => 'textarea',
					'section' => 'hellix_404',
					'label'   => esc_html__( 'Subtitle', 'hellix' ),
				]
			);
			$wp_customize->selective_refresh->add_partial(
				'404_subtitle',
				[
					'render_callback' => function () {
						return esc_html( get_theme_mod( '404_subtitle' ) );
					},
				]
			);

			$wp_customize->add_setting(
				'404_link_text',
				[
					'default'           => esc_html__( 'Back to home', 'hellix' ),
					'sanitize_callback' => 'wp_kses_post',
					'transport'         => 'postMessage',
				]
			);

			$wp_customize->add_control(
				'404_link_text',
				[
					'type'        => 'text',
					'section'     => 'hellix_404',
					'label'       => esc_html__( '404 Link Text', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to change the link text', 'hellix' ),

				]
			);

			$wp_customize->selective_refresh->add_partial(
				'404_link_text',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'button_varient',
				[
					'default'           => 'accent',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'button_varient',
				[
					'type'        => 'select',
					'section'     => 'hellix_404',
					'label'       => esc_html__( 'Button Variation', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to choose your button variations.', 'hellix' ),
					'choices'     => [
						'white'     => esc_html__( 'White', 'hellix' ),
						'secondary' => esc_html__( 'Secondary', 'hellix' ),
						'dark'      => esc_html__( 'Dark', 'hellix' ),
						'accent'    => esc_html__( 'Accent', 'hellix' ),
						'grey'      => esc_html__( 'Grey', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'button_varient',
				[
					'fallback_refresh' => true,
				]
			);
		}

		/**
		 * Customize site footer
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_footer( $wp_customize ) {
			$wp_customize->add_section(
				'hellix_footer',
				[
					'title'       => esc_html__( 'Footer', 'hellix' ),
					'description' => esc_html__( 'Customize the theme footer.', 'hellix' ),
					'priority'    => 90,
				]
			);

			$this->add_footer_section( $wp_customize );
		}

		/**
		 * Adding Footer section controls.
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		private function add_footer_section( $wp_customize ) {
			$this->static_contents = hellix_static_content_options();

			/*
			 * Footer Static Content Enable / Disble Toggle.
			 */
			$wp_customize->add_setting(
				'hellix_footer_static_content',
				[
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'hellix_footer_static_content',
				[
					'type'        => 'select',
					'section'     => 'hellix_footer',
					'label'       => esc_html__( 'Footer Type', 'hellix' ),
					'description' => esc_html__(
						'This settings allow you to choose the footer type.',
						'hellix'
					),
					'choices'     => [
						'no'  => esc_html__( 'Default', 'hellix' ),
						'yes' => esc_html__( 'Static Footer', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'hellix_footer_static_content',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'footer_static_content',
				[
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				]
			);
			$wp_customize->add_control(
				'footer_static_content',
				[
					'type'            => 'select',
					'section'         => 'hellix_footer',
					'label'           => esc_html__( 'Footer Static Content', 'hellix' ),
					'description'     => esc_html__( 'This setting allows you to choose your footer type.', 'hellix' ),
					'choices'         => $this->static_contents,
					'active_callback' => function () {
						return 'yes' === get_theme_mod( 'hellix_footer_static_content', 'no' );
					},
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'footer_static_content',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'footer_skin',
				[
					'default'           => 'dark',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'footer_skin',
				[
					'type'        => 'select',
					'section'     => 'hellix_footer',
					'label'       => esc_html__( 'Footer Variant', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to choose your footer varaince.', 'hellix' ),
					'choices'     => [
						'white' => esc_html__( 'White', 'hellix' ),
						'dark'  => esc_html__( 'Dark', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'footer_skin',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'footer_logo_variant',
				[
					'default'           => 'site_logo',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'footer_logo_variant',
				[
					'type'        => 'select',
					'section'     => 'hellix_footer',
					'label'       => esc_html__( 'Site Logo Variant', 'hellix' ),
					'description' => esc_html__( 'This setting allows you to choose your Logo type.', 'hellix' ),
					'choices'     => [
						'site_logo'   => esc_html__( 'Site Logo', 'hellix' ),
						'site_icon'   => esc_html__( 'Site Icon', 'hellix' ),
						'custom_logo' => esc_html__( 'Custom Logo', 'hellix' ),
					],
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'footer_logo_variant',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'footer_custom_logo_url',
				[
					'fallback_refresh' => true,
				]
			);

			$wp_customize->add_setting(
				'footer_custom_logo_url',
				array(
					'transport'         => 'postMessage',
					'sanitize_callback' => 'absint',
				)
			);

			$wp_customize->add_control(
				new WP_Customize_Media_Control(
					$wp_customize,
					'footer_custom_logo_url',
					array(
						'section'         => 'hellix_footer',
						'label'           => esc_html__( 'Custom Logo Upload', 'hellix' ),
						'description'     => esc_html__(
							'This setting allows you to upload an image for Site Custom Logo.',
							'hellix'
						),
						'mime_type'       => 'image',
						'active_callback' => function () {
							return 'custom_logo' === get_theme_mod( 'footer_logo_variant', 'no' );
						},
					)
				)
			);

			$wp_customize->add_setting(
				'hellix_copyright',
				[
					'transport'         => 'postMessage',
					'sanitize_callback' => 'wp_kses_post',
				]
			);
			$wp_customize->add_control(
				'hellix_copyright',
				[
					'type'        => 'textarea',
					'section'     => 'hellix_footer',
					/* translators: label field for setting in Customizer */
					'label'       => esc_html__( 'Copyright', 'hellix' ),
					/* translators: description field for "Copyright" setting in Customizer */
					'description' => esc_html__( 'HTML is allowed in this field.', 'hellix' ),
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'hellix_copyright',
				[
					'selector'        => '.hellix-copyright',
					'render_callback' => 'hellix_footer_copyright',
				]
			);

			$wp_customize->add_setting(
				'hellix_credit_text',
				[
					'transport'         => 'postMessage',
					'sanitize_callback' => 'wp_kses_post',
				]
			);
			$wp_customize->add_control(
				'hellix_credit_text',
				[
					'type'        => 'textarea',
					'section'     => 'hellix_footer',
					/* translators: label field for setting in Customizer */
					'label'       => esc_html__( 'Credit Text', 'hellix' ),
					/* translators: description field for "Credit Text" setting in Customizer */
					'description' => esc_html__( 'HTML is allowed in this field.', 'hellix' ),
				]
			);

			$wp_customize->selective_refresh->add_partial(
				'hellix_credit_text',
				[
					'selector'        => '.hellix-credit_text',
					'render_callback' => 'hellix_footer_credit_text',
				]
			);
		}

		/**
		 * Returns an array of the desired default Landkit Options
		 *
		 * @return array
		 */
		public function get_hellix_default_setting_values() {
			return apply_filters(
				'hellix_setting_default_values',
				$args = array(
					'hellix_header_button_background' => '#cf574b',
				)
			);
		}

		/**
		 * Adds a value to each Landkit setting if one isn't already present.
		 *
		 * @uses get_landkit_default_setting_values()
		 */
		public function default_theme_mod_values() {
			foreach ( $this->get_hellix_default_setting_values() as $mod => $val ) {
				add_filter( 'theme_mod_' . $mod, array( $this, 'get_theme_mod_value' ), 10 );
			}
		}

		/**
		 * Get theme mod value.
		 *
		 * @param string $value Theme modification value.
		 * @return string
		 */
		public function get_theme_mod_value( $value ) {
			$key = substr( current_filter(), 10 );

			$set_theme_mods = get_theme_mods();

			if ( isset( $set_theme_mods[ $key ] ) ) {
				return $value;
			}

			$values = $this->get_hellix_default_setting_values();

			return isset( $values[ $key ] ) ? $values[ $key ] : $value;
		}

		/**
		 * Set Customizer setting defaults.
		 * These defaults need to be applied separately as child themes can filter landkit_setting_default_values
		 *
		 * @param  array $wp_customize the Customizer object.
		 * @uses   get_landkit_default_setting_values()
		 */
		public function edit_default_customizer_settings( $wp_customize ) {
			foreach ( $this->get_hellix_default_setting_values() as $mod => $val ) {
				$wp_customize->get_setting( $mod )->default = $val;
			}
		}


		/**
		 * Get all of the Landkit theme mods.
		 *
		 * @return array $landkit_theme_mods The Landkit Theme Mods.
		 */
		public function get_hellix_theme_mods() {
			$hellix_theme_mods = array(
				'header_button_background' => get_theme_mod( 'hellix_header_button_background' ),
			);

			return apply_filters( 'hellix_theme_mods', $hellix_theme_mods );
		}

		/**
		 * Get Customizer css.
		 *
		 * @see get_landkit_theme_mods()
		 * @return array $styles the css
		 */
		public function get_css() {
			$hellix_theme_mods       = $this->get_hellix_theme_mods();
			$button_color            = $hellix_theme_mods['header_button_background'];
			$button_color_yiq        = hellix_sass_yiq( $button_color );
			$button_color_darken_10p = hellix_adjust_color_brightness( $button_color, -10 );

			$styles = '
            .btn-primary {
                background-color: ' . $button_color . ';
                color: ' . $button_color_yiq . ';
            }

            .btn-primary:hover,
            .btn-primary:focus {
                 background-color: ' . $button_color_darken_10p . ';
                 color: ' . $button_color_yiq . ';
            }';

			return apply_filters( 'hellix_customizer_css', $styles );
		}

		/**
		 * Add CSS in <head> for styles handled by the theme customizer
		 *
		 * @since 1.0.0
		 * @return void
		 */
		public function add_customizer_css() {
			if ( get_theme_mod( 'button_color' ) === 'custom' ) {
				wp_add_inline_style( 'hellix-style', $this->get_css() );
			}
		}

		/**
		 * Customize site Custom Theme Color
		 *
		 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
		 */
		public function customize_customcolor( $wp_customize ) {
			/*
			 * Custom Color Enable / Disble Toggle.
			 */
			$wp_customize->add_setting(
				'hellix_enable_custom_color',
				[
					'default'           => 'no',
					'sanitize_callback' => 'sanitize_key',
				]
			);

			$wp_customize->add_control(
				'hellix_enable_custom_color',
				[
					'type'        => 'radio',
					'section'     => 'colors',
					'label'       => esc_html__( 'Enable Custom Color?', 'hellix' ),
					'description' => esc_html__(
						'This settings allow you to apply your custom color option.',
						'hellix'
					),
					'choices'     => [
						'yes' => esc_html__( 'Yes', 'hellix' ),
						'no'  => esc_html__( 'No', 'hellix' ),
					],
				]
			);

			/**
			 * Primary Color
			 */
			$wp_customize->add_setting(
				'hellix_primary_color',
				array(
					'default'           => apply_filters( 'hellix_default_primary_color', '#090761' ),
					'sanitize_callback' => 'sanitize_hex_color',
				)
			);
			$wp_customize->add_control(
				new WP_Customize_Color_Control(
					$wp_customize,
					'hellix_primary_color',
					array(
						'label'           => esc_html__( 'Primary color', 'hellix' ),
						'section'         => 'colors',
						'settings'        => 'hellix_primary_color',
						'active_callback' => function () {
							return get_theme_mod( 'hellix_enable_custom_color', 'no' ) === 'yes';
						},
					)
				)
			);

		}


	}
endif;

return new hellix_Customizer();
