<?php
/**
 * Template Functions related to Blog
 */

if ( ! function_exists( 'hellix_posts_sidebar' ) ) {
	/**
	 * Display the posts sidebar.
	 *
	 * @since 1.0.0
	 */
	function hellix_posts_sidebar() {
		$sidebar = get_theme_mod( 'blog_sidebar', 'right-sidebar' );

		if ( ! is_active_sidebar( 'blog-sidebar' ) ) {
			$sidebar = 'no-sidebar';
		}

		return sanitize_key( apply_filters( 'hellix_posts_sidebar', $sidebar ) );
	}
}

if ( ! function_exists( 'hellix_posts_layout' ) ) {
	/**
	 * Display the post layout.
	 *
	 * @since 1.0.0
	 */
	function hellix_posts_layout() {
		$layout = get_theme_mod( 'blog_layout', 'list' );
		return sanitize_key( apply_filters( 'hellix_posts_layout', $layout ) );
	}
}

if ( ! function_exists( 'hellix_post_get_categories' ) ) {
	/**
	 * Display the post get categories.
	 *
	 * @since 1.0.0
	 * @param int $post_id id for post.
	 */
	function hellix_post_get_categories( $post_id = null ) {

		if ( null === $post_id ) {
			$post_id = get_the_ID();
		}

		$categories = get_the_terms( $post_id, 'category' );
		if ( empty( $categories ) || is_wp_error( $categories ) ) {
			return [];
		}

		return $categories;
	}
}

if ( ! function_exists( 'hellix_loop_post_author' ) ) {
	/**
	 * Display the loop post author.
	 *
	 * @since 1.0.0
	 */
	function hellix_loop_post_author() {
		$author = get_the_author(); ?>
		<a class="media meta-link font-size-sm align-items-center pt-3" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
			<?php
			echo wp_kses_post( get_avatar( get_the_author_meta( 'user_email' ), 36, '', '', array( 'class' => 'avatar-img rounded-circle' ) ) );
			if ( ! empty( $author ) ) :
				?>
				<div class="media-body pl-2 ml-1 mt-n1">
				<?php
					echo wp_kses_post(
						sprintf(
							/* translators: %s: author name */
							__( 'by %s', 'hellix' ),
							'<span class="font-weight-semibold ml-1">' . get_the_author() . '</span>'
						)
					);
				?>
				</div>
			<?php endif; ?>
		</a>
		<?php
	}
}

if ( ! function_exists( 'hellix_loop_paging_nav_wrap_start' ) ) {
	/**
	 * Display the loop paging nav start.
	 *
	 * @since 1.0.0
	 */
	function hellix_loop_paging_nav_wrap_start() {
		?>
		<div class="row justify-content-center mt-80 md:mt-32">
			<div class="col-auto">
					<div class="pagination -section">
		<?php
	}
}

if ( ! function_exists( 'hellix_loop_paging_nav_wrap_end' ) ) {
	/**
	 * Display the loop paging nav end.
	 *
	 * @since 1.0.0
	 */
	function hellix_loop_paging_nav_wrap_end() {
		?>
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'hellix_set_blog_posts_per_page' ) ) {
	/**
	 * Display the bolg post per page.
	 *
	 * @since 1.0.0
	 * @param array $query Wp_query object.
	 */
	function hellix_set_blog_posts_per_page( $query ) {
		if ( ! is_admin() && $query->is_main_query() && ! $query->get( 'post_type' ) ) {
			$per_page = $query->get( 'posts_per_page' );
			if ( isset( $_REQUEST['ppp'] ) ) :
				$per_page = intval( $_REQUEST['ppp'] );
				setcookie( 'posts_per_page', intval( $_REQUEST['ppp'] ), time() + 86400, '/' );
			elseif ( isset( $_COOKIE['posts_per_page'] ) ) :
				$per_page = intval( $_COOKIE['posts_per_page'] );
			endif;

			$query->set( 'posts_per_page', $per_page );
		}
	}
}
add_action( 'pre_get_posts', 'hellix_set_blog_posts_per_page' );

if ( ! function_exists( 'hellix_results_count' ) ) {
	/**
	 * Display result count.
	 */
	function hellix_results_count() {
		global $wp_query;

		$total       = $wp_query->found_posts;
		$total_pages = $wp_query->max_num_pages;
		$per_page    = $wp_query->get( 'posts_per_page' );
		$current     = max( 1, $wp_query->get( 'paged', 1 ) );
		?>

		<div class="results-count mb-4 text-center text-md-left">
		<?php
            // phpcs:disable WordPress.Security
		if ( 1 === intval( $total ) ) {
			_e( 'Showing the single post', 'hellix' );
		} elseif ( $total <= $per_page || -1 === $per_page ) {
			/* translators: %d: total posts */
			printf( _n( 'Showing all %d post', 'Showing all %d posts', $total, 'hellix' ), $total );
		} else {
			$first = ( $per_page * $current ) - $per_page + 1;
			$last  = min( $total, $per_page * $current );
			/* translators: 1: first post 2: last post 3: total posts */
			printf( _nx( 'Showing %1$d&ndash;%2$d of %3$d post', 'Showing %1$d&ndash;%2$d of %3$d posts', $total, 'with first and last post', 'hellix' ), $first, $last, $total );
		}
            // phpcs:enable WordPress.Security
		?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'hellix_edit_comment_link' ) ) {
	/**
	 * Display edited comment link.
	 *
	 * @param string $link comment link.
	 * @param int    $comment_comment_id comment id.
	 * @param sting  $text text commented.
	 */
	function hellix_edit_comment_link( $link, $comment_comment_id, $text ) {
		return str_replace(
			[
				'comment-edit-link',
				'\'>',
			],
			[
				'comment-edit-link btn btn-outline-primary btn-sm ml-1',
				'\'>',
			],
			$link
		);
	}
}

/**
 * Outputs the handheld toolbar
 *
 * @since 1.0.0
 */
if ( ! function_exists( 'hellix_handheld_toolbar' ) ) {
	/**
	 * Display hellix handheld toolbar.
	 */
	function hellix_handheld_toolbar() {
		/**
		 * Display tools in toolbar
		 */
		do_action( 'hellix_handheld_toolbar' );
	}
}

/**
 * Outputs the sidebar toggle in the handheld toolbar
 *
 * This toggle should be only the blog page.
 *
 * @since 1.0.0
 */

if ( ! function_exists( 'hellix_handheld_toolbar_toggle_blog_sidebar' ) ) {
	/**
	 * Display hellix_handheld_toolbar_toggle_blog_sidebar.
	 */
	function hellix_handheld_toolbar_toggle_blog_sidebar() {
		if ( ( is_home() || is_singular( 'post' ) || ( 'post' === get_post_type() && ( is_category() || is_tag() || is_author() || is_date() || is_year() || is_month() || is_time() ) ) )
			&& 'no-sidebar' !== hellix_posts_sidebar()
		) :
			?>
		<button class="btn btn-primary btn-sm cs-sidebar-toggle" type="button" data-toggle="offcanvas" data-offcanvas-id="blog-sidebar"><i class="fe-sidebar font-size-base mr-2"></i><?php echo esc_html_x( 'Sidebar', 'front-end', 'hellix' ); ?></button>
			<?php
		endif;
	}
}

if ( ! function_exists( 'hellix_get_blog_style' ) ) {
	/**
	 * Gets Blog page style.
	 *
	 * @return string
	 */
	function hellix_get_blog_style() {
		$layout = get_theme_mod( 'blog_layout', 'list' );
		return apply_filters( 'hellix_blog_style', $layout );
	}
}

if ( ! function_exists( 'hellix_get_blog_layout' ) ) {
	/**
	 * Gets Blog page layout.
	 *
	 * @return string
	 */
	function hellix_get_blog_layout() {
		$style = hellix_get_blog_style();
		if ( 'grid' === $style ) {
			$layout = 'fullwidth';
		} else {
			$layout = get_theme_mod( 'blog_sidebar', 'right-sidebar' );
		}

		return $layout;
	}
}

if ( ! function_exists( 'hellix_blog_has_sidebar' ) ) {
	/**
	 * Checks if the blog page has a sidebar or not.
	 *
	 * @return bool
	 */
	function hellix_blog_has_sidebar() {
		$has_sidebar = false;
		$layout      = hellix_posts_sidebar();

		if ( 'left-sidebar' === $layout || 'right-sidebar' === $layout ) {
			if ( is_active_sidebar( 'blog-sidebar' ) ) {
				$has_sidebar = true;
			}
		}

		return $has_sidebar;
	}
}


if ( ! function_exists( 'hellix_comments_navigation' ) ) {
	/**
	 * Comments navigation.
	 */
	function hellix_comments_navigation() {
		if ( absint( get_comment_pages_count() ) === 1 ) {
			return;
		}

		/* translators: label for link to the previous comments page */
		$prev_text = esc_html__( 'Older comments', 'hellix' );
		$prev_link = get_previous_comments_link( '<i class="fe fe-arrow-left mr-2"></i>' . $prev_text );

		/* translators: label for link to the next comments page */
		$next_text = esc_html__( 'Newer comments', 'hellix' );
		$next_link = get_next_comments_link( $next_text . '<i class="fe fe-arrow-right ml-2"></i>' );

		?>
		<nav class="navigation comment-navigation d-flex justify-content-between my-5" role="navigation">
			<h3 class="screen-reader-text sr-only">
			<?php
			/* translators: navigation through comments */
			echo esc_html__( 'Comment navigation', 'hellix' );
			?>
			</h3>
			<?php if ( $prev_link ) : ?>
				<?php echo str_replace( '<a ', '<a class="text-decoration-none font-weight-medium" ', $prev_link ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
			<?php endif; ?>
			<?php if ( $next_link ) : ?>
				<?php echo str_replace( '<a ', '<a class="text-decoration-none ml-auto font-weight-medium" ', $next_link ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
			<?php endif; ?>
		</nav>
		<?php
	}
}
