<?php
/**
 * Hellix WooCommerce Class
 *
 * @package  hellix
 * @since    0.0.1
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Hellix_WooCommerce' ) ) :

	/**
	 * The Hellix WooCommerce Integration class
	 */
	class Hellix_WooCommerce {

		/**
		 * Setup class.
		 *
		 * @since 1.0
		 */
		public function __construct() {
			$this->init_hooks();
		}


		/**
		 * Setup class.
		 *
		 * @since 1.0
		 */
		private function init_hooks() {
			add_action( 'widgets_init', array( $this, 'widgets_init' ), 10 );
			add_action( 'after_setup_theme', array( $this, 'setup' ) );
			add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );
			add_filter( 'woocommerce_post_class', array( $this, 'product_class' ), 10 );
			add_filter( 'woocommerce_product_description_heading', '__return_false' );
			add_filter( 'woocommerce_product_additional_information_heading', '__return_false' );
			add_filter( 'hellix_is_page_prose_enabled', array( $this, 'set_not_prose' ) );
			add_filter( 'hellix_page_header_bg_image', array( $this, 'wc_header_bg_image' ) );
			add_filter( 'hellix_wc_page_header_title', array( $this, 'override_title' ) );
			add_filter( 'hellix_wc_page_header_subtitle', array( $this, 'override_subtitle' ) );
		}

		/**
		 * Override title.
		 *
		 * @param string $title Title of the masthead.
		 * @return string
		 */
		public function override_title( $title ) {
			if ( is_shop() ) {
				$title = get_theme_mod( 'header_title', 'Shop' );
			} elseif ( is_product() ) {
				$title = get_the_title();
			}

			return $title;
		}

		/**
		 * Override subtitle.
		 *
		 * @param string $subtitle Subtitle of the masthead.
		 * @return string
		 */
		public function override_subtitle( $subtitle ) {
			if ( is_shop() || is_product() ) {
				$subtitle = get_theme_mod( 'header_subtitle', 'Products' );
			} elseif ( is_cart() || is_checkout() ) {
				$subtitle = get_theme_mod( 'header_title', 'Shop' );
			}

			return $subtitle;
		}

		/**
		 * Override BG image for WooCommerce Pages.
		 *
		 * @param string $header_bg_image URL of the BG Image.
		 * @return string
		 */
		public function wc_header_bg_image( $header_bg_image ) {
			$header_image      = get_template_directory_uri() . '/assets/img/backgrounds/13.jpg';
			$shop_header_image = wp_get_attachment_url( get_theme_mod( 'shop_header_image', $header_image ) );

			if ( is_woocommerce() || is_account_page() || is_cart() || is_checkout() ) {
				if ( empty( $shop_header_image ) ) {
					$header_bg_image = get_header_image();
				} else {
					$header_bg_image = $shop_header_image;
				}
			}
			return $header_bg_image;
		}

		/**
		 * Set not prose for WooCommerce Pages.
		 *
		 * @param bool $enabled Prose enabled or disabled.
		 * @return bool
		 */
		public function set_not_prose( $enabled ) {
			if ( is_cart() || is_checkout() || is_account_page() ) {
				$enabled = false;
			}
			return $enabled;
		}

		/**
		 * Sets up theme defaults and registers support for various WooCommerce features.
		 *
		 * Note that this function is hooked into the after_setup_theme hook, which
		 * runs before the init hook. The init hook is too late for some features, such
		 * as indicating support for post thumbnails.
		 *
		 * @since 1.0.0
		 * @return void
		 */
		public function widgets_init() {

			$sidebar_args = $this->get_wc_shop_sidebar_args();

			foreach ( $sidebar_args as $sidebar => $args ) {
				$widget_tags = array(
					'before_widget' => '<div class="col-lg-12 col-md-6"><div id="%1$s" class="shopSidebar__item %2$s">',
					'after_widget'  => '</div></div>',
					'before_title'  => '<h4 class="uppercase text-lg mb-24 leading-md">',
					'after_title'   => '</h4>',
				);

				$filter_hook = sprintf( 'hellix_%s_widget_tags', $sidebar );
				$widget_tags = apply_filters( $filter_hook, $widget_tags );

				if ( is_array( $widget_tags ) ) {
					register_sidebar( $args + $widget_tags );
				}
			}
		}

		/**
		 * Getting blog sidebar arguments.
		 */
		public function get_wc_shop_sidebar_args() {
			$sidebar_args['shop_sidebar'] = array(
				'name'        => esc_html__( 'Shop Sidebar', 'hellix' ),
				'id'          => 'sidebar-shop',
				'description' => '',
			);

			return apply_filters( 'hellix_wc_shop_sidebar_args', $sidebar_args );
		}
		 /**
		  * Sets up theme defaults and registers support for various WooCommerce features.
		  *
		  * Note that this function is hooked into the after_setup_theme hook, which
		  * runs before the init hook. The init hook is too late for some features, such
		  * as indicating support for post thumbnails.
		  *
		  * @since 1.0.0
		  * @return void
		  */
		public function setup() {
			// Declare WooCommerce support.
			add_theme_support(
				'woocommerce',
				apply_filters(
					'hellix_woocommerce_args',
					array(
						'single_image_width'    => 644,
						'thumbnail_image_width' => 350,
						'product_grid'          => array(
							'default_columns' => 4,
							'default_rows'    => 4,
							'min_columns'     => 1,
							'max_columns'     => 6,
							'min_rows'        => 1,
						),
					)
				)
			);

			add_theme_support( 'wc-product-gallery-zoom' );
			add_theme_support( 'wc-product-gallery-lightbox' );
			add_theme_support( 'wc-product-gallery-slider' );

			/**
			 * Add 'hellix_woocommerce_setup' action.
			 *
			 * @since  1.0.0
			 */
			do_action( 'hellix_woocommerce_setup' );

		}

		/**
		 * Function to add classes to product.
		 *
		 * @param string $classes the class to be added to product.
		 */
		public function product_class( $classes ) {
			$columns = apply_filters( 'hellix_loop_shop_columns', wc_get_loop_prop( 'columns' ) );

			$classes[] = 'col-lg-' . esc_attr( $columns ) . '  col-sm-6';

			return $classes;

		}

	}

endif;

return new Hellix_WooCommerce();
