<?php
/**
 * Hellix Class
 *
 * @package  hellix
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'Hellix' ) ) :

	/**
	 * The main Hellix class
	 */
	class Hellix {

		/**
		 * Setup class
		 */
		public function __construct() {
			add_action( 'after_setup_theme', array( $this, 'setup' ) );
			add_action( 'widgets_init', array( $this, 'widgets_init' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ), 10 );
			add_action( 'wp_enqueue_scripts', array( $this, 'child_scripts' ), 30 ); // After WooCommerce.
			add_filter( 'body_class', array( $this, 'body_classes' ) );
			add_action( 'wp_head', array( $this, 'prefetch' ) );
		}

		/**
		 * Register Google fonts.
		 *
		 * @return string Google fonts URL for the theme.
		 */
		public function google_fonts() {
			$fonts_url = 'https://fonts.googleapis.com/css2?family=Work+Sans:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap';
			return $fonts_url;
		}

		/**
		 * Sets up theme defaults and registers support for various WordPress features.
		 *
		 * Note that this function is hooked into the after_setup_theme hook, which
		 * runs before the init hook. The init hook is too late for some features, such
		 * as indicating support for post thumbnails.
		 */
		public function setup() {
			/*
			 * Load Localisation files.
			 *
			 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
			 */

			// Loads wp-content/languages/themes/hellix-it_IT.mo.
			load_theme_textdomain( 'hellix', trailingslashit( WP_LANG_DIR ) . 'themes' );

			// Loads wp-content/themes/child-theme-name/languages/it_IT.mo.
			load_theme_textdomain( 'hellix', get_stylesheet_directory() . '/languages' );

			// Loads wp-content/themes/hellix/languages/it_IT.mo.
			load_theme_textdomain( 'hellix', get_template_directory() . '/languages' );

			/**
			 * Enqueue editor styles.
			 */
			add_theme_support( 'editor-styles' );
			add_editor_style( array( 'assets/css/gutenberg-editor.css', $this->google_fonts() ) );

			/**
			 * Add default posts and comments RSS feed links to head.
			 */
			add_theme_support( 'automatic-feed-links' );

			/*
			 * Enable support for Post Thumbnails on posts and pages.
			 *
			 * @link https://developer.wordpress.org/reference/functions/add_theme_support/#Post_Thumbnails
			 */
			add_theme_support( 'post-thumbnails' );

			/**
			 * Enable support for site logo.
			 */
			$custom_logo_args = apply_filters(
				'hellix_custom_logo_args',
				array(
					'height'      => 50,
					'width'       => 48,
					'flex-width'  => true,
					'flex-height' => true,
				)
			);
			add_theme_support( 'custom-logo', $custom_logo_args );

			/**
			 * Register menu locations.
			 */
			register_nav_menus(
				apply_filters(
					'hellix_register_nav_menus',
					array(
						'primary'      => esc_html__( 'Primary Menu', 'hellix' ),
						'secondary'    => esc_html__( 'Secondary Menu', 'hellix' ),
						'social'       => esc_html__( 'Social Media Menu', 'hellix' ),
						'topbar'       => esc_html__( 'Topbar Menu', 'hellix' ),
						'footer_links' => esc_html_x( 'Footer Menu', 'menu location', 'hellix' ),
					)
				)
			);

			/*
			 * Switch default core markup for search form, comment form, comments, galleries, captions and widgets
			 * to output valid HTML5.
			 */
			$html5_args = apply_filters(
				'hellix_html5_args',
				array(
					'search-form',
					'comment-form',
					'comment-list',
					'gallery',
					'caption',
					'widgets',
					'style',
					'script',
				)
			);
			add_theme_support( 'html5', $html5_args );

			/**
			 *  Add support for the Site Logo plugin and the site logo functionality in JetPack
			 *  https://github.com/automattic/site-logo
			 *  http://jetpack.me/
			 */
			add_theme_support(
				'site-logo',
				apply_filters(
					'hellix_site_logo_args',
					array(
						'size' => 'full',
					)
				)
			);

			/**
			 * Setup the WordPress core custom header feature.
			 */
			$custom_header_args = apply_filters(
				'hellix_custom_header_args',
				array(
					'default-image' => get_template_directory_uri() . '/assets/img/backgrounds/12.jpg',
					'header-text'   => false,
					'width'         => 1950,
					'height'        => 500,
					'flex-width'    => true,
					'flex-height'   => true,
				)
			);
			add_theme_support( 'custom-header', $custom_header_args );

			/**
			 * Declare support for title theme feature.
			 */
			add_theme_support( 'title-tag' );

			/**
			 * Add support for responsive embedded content.
			 */
			add_theme_support( 'responsive-embeds' );
		}

		/**
		 * Getting blog sidebar arguments.
		 */
		public function get_blog_sidebar_args() {
			$blog_sidebar_args['blog_sidebar'] = array(
				'name'        => esc_html__( 'Blog Sidebar', 'hellix' ),
				'id'          => 'blog-sidebar',
				'description' => '',
			);

			return apply_filters( 'hellix_blog_sidebar_args', $blog_sidebar_args );
		}

		/**
		 * Getting blog sidebar arguments.
		 */
		public function get_canvas_sidebar_args() {
			$canvas_sidebar_args['canvas_sidebar'] = array(
				'name'        => esc_html__( 'Off Canvas Sidebar', 'hellix' ),
				'id'          => 'canvas-sidebar',
				'description' => '',
			);

			return apply_filters( 'hellix_canvas_sidebar_args', $canvas_sidebar_args );
		}

		/**
		 * Register widget area.
		 *
		 * @link https://codex.wordpress.org/Function_Reference/register_sidebar
		 */
		public function widgets_init() {

			$blog_sidebar_args   = $this->get_blog_sidebar_args();
			$canvas_sidebar_args = $this->get_canvas_sidebar_args();

			foreach ( $blog_sidebar_args  as $sidebar => $args ) {
				$widget_tags = array(
					'before_widget' => '<div id="%1$s" class="blogSidebar__item %2$s">',
					'after_widget'  => '</div>',
					'before_title'  => '<h5 class="blogSidebar__title">',
					'after_title'   => '</h5>',
				);
				/**
				 * Dynamically generated filter hooks. Allow changing widget wrapper and title tags. See the list below.
				 *
				 * 'bookworm_shop_sidebar_widget_tags'
				 */
				$filter_hook = sprintf( 'hellix_%s_widget_tags', $sidebar );
				$widget_tags = apply_filters( $filter_hook, $widget_tags );

				if ( is_array( $widget_tags ) ) {
					register_sidebar( $args + $widget_tags );
				}
			}

			foreach ( $canvas_sidebar_args  as $sidebar => $args ) {
				$widget_tags = array(
					'before_widget' => '',
					'after_widget'  => '',
				);

				$filter_hook = sprintf( 'hellix_%s_widget_tags', $sidebar );
				$widget_tags = apply_filters( $filter_hook, $widget_tags );

				if ( is_array( $widget_tags ) ) {
					register_sidebar( $args + $widget_tags );
				}
			}
		}

		/**
		 * Enqueue scripts and styles.
		 */
		public function scripts() {
			global $hellix_version;

			/**
			 * Fonts
			 */
			wp_enqueue_style( 'hellix-google-font', 'https://fonts.googleapis.com/css2?family=Work+Sans:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900&display=swap', '', $hellix_version );
			wp_enqueue_style( 'hellix-google-material-font', 'https://fonts.googleapis.com/css2?family=Material+Icons+Outlined', '', $hellix_version );

			/**
			 * Styles
			 */
			if ( ! apply_filters( 'hellix_load_min_css', false ) ) {
				wp_enqueue_style( 'hellix-bootstrap', get_template_directory_uri() . '/assets/css/vendors/bootstrap.css', '', $hellix_version );
				wp_enqueue_style( 'hellix-fontawesome', get_template_directory_uri() . '/assets/css/vendors/fontawesome.css', '', $hellix_version );
				wp_enqueue_style( 'hellix-lightbox', get_template_directory_uri() . '/assets/css/vendors/lightbox.css', '', $hellix_version );
				wp_enqueue_style( 'hellix-icomoon', get_template_directory_uri() . '/assets/css/vendors/icomoon.css', '', $hellix_version );
				wp_enqueue_style( 'hellix-swiper', get_template_directory_uri() . '/assets/css/vendors/swiper.css', '', $hellix_version );
			} else {
				wp_enqueue_style( 'hellix-vendors-css', get_template_directory_uri() . '/assets/css/vendors.css', '', $hellix_version );
			}

			wp_enqueue_style( 'hellix-style', get_template_directory_uri() . '/style.css', '', $hellix_version );

			/**
			 * Scripts
			 */
			if ( ! apply_filters( 'hellix_load_min_js', false ) ) {
				wp_enqueue_script( 'hellix-custom-ease-js', get_template_directory_uri() . '/assets/js/vendors/custom-ease.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-drawsvgplugin-js', get_template_directory_uri() . '/assets/js/vendors/drawsvgplugin.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-gsap-js', get_template_directory_uri() . '/assets/js/vendors/gsap.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-headroom-js', get_template_directory_uri() . '/assets/js/vendors/headroom.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-isotope-js', get_template_directory_uri() . '/assets/js/vendors/isotope.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-jarallax-js', get_template_directory_uri() . '/assets/js/vendors/jarallax.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-popper-js', get_template_directory_uri() . '/assets/js/vendors/popper.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-scroll-magic-js', get_template_directory_uri() . '/assets/js/vendors/scroll-magic.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-scrolltoplugin-js', get_template_directory_uri() . '/assets/js/vendors/scrolltoplugin.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-split-text-js', get_template_directory_uri() . '/assets/js/vendors/split-text.js', array(), $hellix_version, true );
				wp_enqueue_script( 'hellix-swiper-js', get_template_directory_uri() . '/assets/js/vendors/swiper.js', array(), $hellix_version, true );
			} else {
				wp_enqueue_script( 'hellix-vendors', get_template_directory_uri() . '/assets/js/vendors.js', array(), $hellix_version, true );
			}

			wp_enqueue_script( 'hellix-main', get_template_directory_uri() . '/assets/js/main.js', array(), $hellix_version, true );

			if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
				wp_enqueue_script( 'comment-reply' );
			}

			$admin_ajax_url = admin_url( 'admin-ajax.php' );
			$current_lang   = apply_filters( 'wpml_current_language', null );
			if ( $current_lang ) {
				$admin_ajax_url = add_query_arg( 'lang', $current_lang, $admin_ajax_url );
			}

			$hellix_options = apply_filters(
				'around_localize_script_data',
				array(
					'ajax_url'        => $admin_ajax_url,
					'ajax_loader_url' => get_template_directory_uri() . '/assets/images/ajax-loader.gif',

				)
			);

			wp_localize_script( 'hellix-main', 'hellix_options', $hellix_options );

		}

		/**
		 * Prefetch
		 */
		public function prefetch() {
			?>
			<link rel="preconnect" href="https://fonts.googleapis.com">
			<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
			<?php
		}

		/**
		 * Enqueue child theme stylesheet.
		 * A separate function is required as the child theme css needs to be enqueued _after_ the parent theme
		 * primary css and the separate WooCommerce css.
		 */
		public function child_scripts() {
			if ( is_child_theme() ) {
				$child_theme = wp_get_theme( get_stylesheet() );
				wp_enqueue_style( 'hellix-child-style', get_stylesheet_uri(), array(), $child_theme->get( 'Version' ) );
			}
		}

		/**
		 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
		 *
		 * @param array $args Configuration arguments.
		 * @return array
		 */
		public function page_menu_args( $args ) {
			$args['show_home'] = true;
			return $args;
		}

		/**
		 * Adds custom classes to the array of body classes.
		 *
		 * @param array $classes Classes for the body element.
		 * @return array
		 */
		public function body_classes( $classes ) {
			$classes[] = 'break-words';
			if ( hellix_is_page_loader() ) {
				$classes[] = 'preloader-visible';
			}
			return $classes;
		}

	}
endif;

return new Hellix();
