<?php
/**
 * Functions that are used as templates.
 *
 * @package hellix
 */

/**
 * Apply inline style to the hellix header.
 *
 * @uses  get_header_image()
 * @since  1.0.0
 */
function hellix_header_styles() {
$is_header_image = apply_filters( 'hellix_page_header_bg_image', get_header_image() );
$header_bg_image = '';
$header_bg       = get_theme_mod( 'masthead_bg_color', '#18130C' );

if ( $is_header_image ) {
$header_bg_image = 'url(' . esc_url( $is_header_image ) . ')';
}

$styles = array();

if ( '' !== $header_bg_image ) {
$styles['background-image'] = $header_bg_image;
} else {
$styles['background-color'] = $header_bg;
}

$styles = apply_filters( 'hellix_header_styles', $styles );

foreach ( $styles as $style => $value ) {
echo esc_attr( $style . ': ' . $value . '; ' );
}
}

/**
 * Template for page masthead
 *
 * @param string $subtitle The subtitle for the masthead.
 */
function hellix_output_page_masthead( $subtitle = '' ) {

	$title           = get_the_title();
	$is_header_image = get_header_image();

	$title           = apply_filters( 'hellix_page_header_title', $title );
	$is_header_image = apply_filters( 'hellix_page_header_bg_image', $is_header_image );
	$subtitle        = apply_filters( 'hellix_page_header_subtitle', '' );

	if ( is_home() && is_front_page() ) {
		$title    = esc_html__( 'Blog', 'hellix' );
		$subtitle = '';
	} elseif ( is_home() ) {
		$blog_page_id = get_option( 'page_for_posts', true );
		$title        = get_the_title( $blog_page_id );
	}

	if ( is_archive() ) {
		$title_arr = explode( ':', get_the_archive_title() );
		if ( isset( $title_arr[1] ) ) {
			$subtitle = $title_arr[0];
			$title    = $title_arr[1];
		} else {
			$title = $title_arr[0];
		}
	}

	if ( is_search() ) {
		$title    = get_search_query();
		$subtitle = esc_html__( 'Search Results for', 'hellix' );
	}

	$title    = apply_filters( 'hellix_wc_page_header_title', $title );
	$subtitle = apply_filters( 'hellix_wc_page_header_subtitle', $subtitle );

	?>
	<section class="page-masthead">
		<div data-parallax="0.6" class="page-masthead__bg">
			<div data-parallax-target class="bg-image js-lazy" style="<?php hellix_header_styles(); ?>"></div>
		</div>
		<div class="container">
			<div class="page-masthead__content">
				<div class="row justify-content-between md:justify-content-center align-items-center">
					<?php if ( ! empty( $subtitle ) || ! empty( $title ) ) : ?>
					<div class="col-lg-9 col-md-10">
						<div data-anim="slide-up delay-1">
							<?php if ( ! empty( $subtitle ) ) : ?>
							<div class="page-masthead__subtitle"><?php echo esc_html( $subtitle ); ?></div>
							<?php endif; ?>
							<?php if ( ! empty( $title ) ) : ?>
							<div class="page-masthead__back_title"><?php echo wp_kses_post( $title ); ?></div>
							<h1 class="page-masthead__title text-white"><?php echo wp_kses_post( $title ); ?></h1>
							<?php endif; ?>
						</div>
					</div>
					<?php endif; ?>
					<div class="col"><?php hellix_breadcrumb(); ?></div>
				</div>
			</div>
		</div>
	</section>
	<?php
}


/**
 * Displays the class names for the main element.
 *
 * @param string|string[] $class Space-separated string or array of class names to add to the class list.
 */
function hellix_main_class( $class = '' ) {
	// Separates class names with a single space, collates class names for main element.
	echo 'class="' . esc_attr( implode( ' ', hellix_get_main_class( $class ) ) ) . '"';
}

/**
 * Retrieves an array of the class names for the main element.
 *
 * @global WP_Query $wp_query WordPress Query object.
 *
 * @param string|string[] $class Space-separated string or array of class names to add to the class list.
 * @return string[] Array of class names.
 */
function hellix_get_main_class( $class = '' ) {
	global $wp_query;

	$classes       = array( 'main-content', 'relative' );
	$version       = hellix_get_header_version();
	$enable_topbar = hellix_header_topbar_inner_content();

	if ( 'v2' === $version ) {
		$classes[] = 'bg-black';
		$classes[] = 'pattern_bg_2';
	}

	if ( 'yes' === $enable_topbar && ( 'v2' === $version || 'v3' === $version ) ) {
		$classes[] = 'has-topbar';
	}
	if ( ! empty( $class ) ) {
		if ( ! is_array( $class ) ) {
			$class = preg_split( '#\s+#', $class );
		}
		$classes = array_merge( $classes, $class );
	} else {
		// Ensure that we always coerce class to being an array.
		$class = array();
	}

	$classes = array_map( 'esc_attr', $classes );

	/**
	 * Filters the list of CSS body class names for the current post or page.
	 *
	 * @since 2.8.0
	 *
	 * @param string[] $classes An array of body class names.
	 */
	$classes = apply_filters( 'hellix_main_class', $classes, $class );

	return array_unique( $classes );
}

/**
 * Displays the class names for the #content element.
 *
 * @param string|string[] $class Space-separated string or array of class names to add to the class list.
 */
function hellix_content_class( $class = '' ) {
	// Separates class names with a single space, collates class names for body element.
	echo 'class="' . esc_attr( implode( ' ', hellix_get_content_class( $class ) ) ) . '"';
}

/**
 * Retrieves an array of the class names for the body element.
 *
 * @global WP_Query $wp_query WordPress Query object.
 *
 * @param string|string[] $class Space-separated string or array of class names to add to the class list.
 * @return string[] Array of class names.
 */
function hellix_get_content_class( $class = '' ) {
	global $wp_query;

	$classes = array( 'content-wrapper', 'js-content-wrapper' );
	$version = hellix_get_header_version();

	if ( 'v10' === $version ) {
		$classes[] = '-left';
	}

	if ( ! empty( $class ) ) {
		if ( ! is_array( $class ) ) {
			$class = preg_split( '#\s+#', $class );
		}
		$classes = array_merge( $classes, $class );
	} else {
		// Ensure that we always coerce class to being an array.
		$class = array();
	}

	$classes = array_map( 'esc_attr', $classes );

	/**
	 * Filters the list of CSS body class names for the current post or page.
	 *
	 * @since 2.8.0
	 *
	 * @param string[] $classes An array of body class names.
	 */
	$classes = apply_filters( 'hellix_content_class', $classes, $class );

	return array_unique( $classes );
}

/**
 * Get Copyright Text
 */
function hellix_get_copyright_text() {
	/* translators: %1$s - Year, %2$s - Name of the blog */
	$text = apply_filters( 'hellix_copyright_text', sprintf( esc_html__( '&copy; %1$s %2$s. All rights reserved', 'hellix' ), gmdate( 'Y' ), get_bloginfo( 'name' ) ) );
	return $text;
}

/**
 * Get Credit Text
 */
function hellix_get_credit_text() {
	$text = apply_filters( 'hellix_credit_text', esc_html__( 'Built by MadrasThemes', 'hellix' ) );
	return $text;
}

/**
 * Get other templates (e.g. product attributes) passing attributes and including the file.
 *
 * @param string $template_name Template name.
 * @param array  $args          Arguments. (default: array).
 * @param string $template_path Template path. (default: '').
 * @param string $default_path  Default path. (default: '').
 */
function hellix_get_template( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
	global $hellix_version;
	$cache_key = sanitize_key( implode( '-', array( 'template', $template_name, $template_path, $default_path, $hellix_version ) ) );
	$template  = (string) wp_cache_get( $cache_key, 'hellix' );

	if ( ! $template ) {
		$template = hellix_locate_template( $template_name, $template_path, $default_path );

		// Don't cache the absolute path so that it can be shared between web servers with different paths.
		$cache_path = hellix_tokenize_path( $template, hellix_get_path_define_tokens() );

		hellix_set_template_cache( $cache_key, $cache_path );
	} else {
		// Make sure that the absolute path to the template is resolved.
		$template = hellix_untokenize_path( $template, hellix_get_path_define_tokens() );
	}

	// Allow 3rd party plugin filter template file from their plugin.
	$filter_template = apply_filters( 'hellix_get_template', $template, $template_name, $args, $template_path, $default_path );

	if ( $filter_template !== $template ) {
		if ( ! file_exists( $filter_template ) ) {
			/* translators: %s template */
			_doing_it_wrong( __FUNCTION__, sprintf( esc_html__( '%s does not exist.', 'hellix' ), '<code>' . esc_html( $filter_template ) . '</code>' ), '2.1' );
			return;
		}
		$template = $filter_template;
	}

	$action_args = array(
		'template_name' => $template_name,
		'template_path' => $template_path,
		'located'       => $template,
		'args'          => $args,
	);

	if ( ! empty( $args ) && is_array( $args ) ) {
		if ( isset( $args['action_args'] ) ) {
			_doing_it_wrong(
				__FUNCTION__,
				esc_html__( 'action_args should not be overwritten when calling hellix_get_template.', 'hellix' ),
				'3.6.0'
			);
			unset( $args['action_args'] );
		}
		extract( $args ); // @codingStandardsIgnoreLine
	}

	do_action( 'hellix_before_template_part', $action_args['template_name'], $action_args['template_path'], $action_args['located'], $action_args['args'] );

	include $action_args['located'];

	do_action( 'hellix_after_template_part', $action_args['template_name'], $action_args['template_path'], $action_args['located'], $action_args['args'] );
}

/**
 * Locate a template and return the path for inclusion.
 *
 * This is the load order:
 *
 * yourtheme/$template_path/$template_name
 * yourtheme/$template_name
 * $default_path/$template_name
 *
 * @param string $template_name Template name.
 * @param string $template_path Template path. (default: '').
 * @param string $default_path  Default path. (default: '').
 * @return string
 */
function hellix_locate_template( $template_name, $template_path = '', $default_path = '' ) {
	if ( ! $template_path ) {
		$template_path = 'templates/';
	}

	if ( ! $default_path ) {
		$default_path = 'templates/';
	}

	// Look within passed path within the theme - this is priority.
	if ( false !== strpos( $template_name, 'product_cat' ) || false !== strpos( $template_name, 'product_tag' ) ) {
		$cs_template = str_replace( '_', '-', $template_name );
		$template    = locate_template(
			array(
				trailingslashit( $template_path ) . $cs_template,
				$cs_template,
			)
		);
	}

	if ( empty( $template ) ) {
		$template = locate_template(
			array(
				trailingslashit( $template_path ) . $template_name,
				$template_name,
			)
		);
	}

	// Get default template/.
	if ( ! $template ) {
		if ( empty( $cs_template ) ) {
			$template = $default_path . $template_name;
		} else {
			$template = $default_path . $cs_template;
		}
	}

	// Return what we found.
	return apply_filters( 'hellix_locate_template', $template, $template_name, $template_path );
}

/**
 * Given a path, this will convert any of the subpaths into their corresponding tokens.
 *
 * @param string $path The absolute path to tokenize.
 * @param array  $path_tokens An array keyed with the token, containing paths that should be replaced.
 * @return string The tokenized path.
 */
function hellix_tokenize_path( $path, $path_tokens ) {
	// Order most to least specific so that the token can encompass as much of the path as possible.
	uasort(
		$path_tokens,
		function ( $a, $b ) {
			$a = strlen( $a );
			$b = strlen( $b );

			if ( $a > $b ) {
				return -1;
			}

			if ( $b > $a ) {
				return 1;
			}

			return 0;
		}
	);

	foreach ( $path_tokens as $token => $token_path ) {
		if ( 0 !== strpos( $path, $token_path ) ) {
			continue;
		}

		$path = str_replace( $token_path, '{{' . $token . '}}', $path );
	}

	return $path;
}

/**
 * Given a tokenized path, this will expand the tokens to their full path.
 *
 * @param string $path The absolute path to expand.
 * @param array  $path_tokens An array keyed with the token, containing paths that should be expanded.
 * @return string The absolute path.
 */
function hellix_untokenize_path( $path, $path_tokens ) {
	foreach ( $path_tokens as $token => $token_path ) {
		$path = str_replace( '{{' . $token . '}}', $token_path, $path );
	}

	return $path;
}

/**
 * Fetches an array containing all of the configurable path constants to be used in tokenization.
 *
 * @return array The key is the define and the path is the constant.
 */
function hellix_get_path_define_tokens() {
	$dir     = 'DIR';
	$defines = array(
		'ABSPATH',
		'WP_CONTENT_DIR',
		'WP_PLUGIN_DIR',
		'WPMU_PLUGIN_DIR',
		'PLUGIN' . $dir,
		'WP_THEME_DIR',
	);

	$path_tokens = array();
	foreach ( $defines as $define ) {
		if ( defined( $define ) ) {
			$path_tokens[ $define ] = constant( $define );
		}
	}

	return apply_filters( 'hellix_get_path_define_tokens', $path_tokens );
}

/**
 * Add a template to the template cache.
 *
 * @param string $cache_key Object cache key.
 * @param string $template Located template.
 */
function hellix_set_template_cache( $cache_key, $template ) {
	wp_cache_set( $cache_key, $template, 'hellix' );

	$cached_templates = wp_cache_get( 'cached_templates', 'hellix' );
	if ( is_array( $cached_templates ) ) {
		$cached_templates[] = $cache_key;
	} else {
		$cached_templates = array( $cache_key );
	}

	wp_cache_set( 'cached_templates', $cached_templates, 'hellix' );
}

/**
 * Clear the template cache.
 */
function hellix_clear_template_cache() {
	$cached_templates = wp_cache_get( 'cached_templates', 'hellix' );
	if ( is_array( $cached_templates ) ) {
		foreach ( $cached_templates as $cache_key ) {
			wp_cache_delete( $cache_key, 'hellix' );
		}

		wp_cache_delete( 'cached_templates', 'hellix' );
	}
}

/**
 * Print pagination links.
 *
 * @param array $pages Page Links.
 * @param array $args  The arguments for generating the pagination links.
 * @return string|null
 */
function hellix_print_pagination_links( $pages, $args ) {

	$pagination       = null;
	$pagination_class = 'pagination -section';
	$total_pages      = $args['total'];
	if ( is_array( $pages ) ) {
		if ( ! empty( $args['pagination_class'] ) ) {
			$pagination_class .= ' ' . $args['pagination_class'];
		}

		$pagination = '<div class="' . esc_attr( $pagination_class ) . '">';
		$paged      = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

		/* translators: %1$s - Current Page, %2$s - Total Pages */
		$paged_string = sprintf( esc_html__( '%1$s / %2$s', 'hellix' ), $paged, $total_pages );
		$page_numbers = false;
		$count        = 1;
		foreach ( $pages as $page ) {
			if ( ( false !== strpos( $page, 'page-numbers prev' ) ) ) {
				$before     = '<div class="pagination__nav">';
				$after      = '</div>';
				$icon       = 'nav-icon -left';
				$icon_class = 'icon icon-left-arrow';

			} elseif ( ( false !== strpos( $page, '"page-numbers"' ) && false === $page_numbers ) || ( false !== strpos( $page, 'current' ) && false === $page_numbers ) ) {
				$before       = '<div class="pagination__content d-flex flex-wrap">';
				$after        = '';
				$icon         = '';
				$icon_class   = '';
				$page_numbers = true;
			} elseif ( false !== strpos( $page, 'page-numbers next' ) ) {
				$before     = true === $page_numbers ? '</div><div class="pagination__nav">' : '<div class="pagination__nav">';
				$after      = '</div>';
				$icon       = 'nav-icon -right';
				$icon_class = 'icon icon-right-arrow';
			} else {
				$before     = '';
				$after      = '';
				$icon       = '';
				$icon_class = '';
			}
			if ( true === $page_numbers && count( $pages ) === $count ) {
				$after = '</div>';
			}

			if ( strpos( $page, 'prev' ) || strpos( $page, 'next' ) ) {
				$page        = str_replace( 'page-numbers', $icon, $page );
				$page        = str_replace( '<a', '<a data-barba', $page );
				$pagination .= $before . str_replace( '></a>', '><div class="nav-icon__circle"><i class="' . esc_attr( $icon_class ) . '"></i></div></a>', $page ) . $after;
			} else {
				if ( ( strpos( $page, 'current' ) !== false ) ) {
					$page        = str_replace( 'span', 'a', $page );
					$page        = str_replace( '<a', '<a data-barba href="#"', $page );
					$pagination .= $before . ( strpos( $page, 'current' ) !== false ? str_replace( 'page-numbers current', 'is-active', $page ) : '' ) . $after;
				} elseif ( ( strpos( $page, 'dots' ) !== false ) ) {
					$pagination .= $before . ( strpos( $page, 'dots' ) !== false ? str_replace( 'page-numbers dots>', '>....', $page ) : '' ) . $after;
				} else {
					$pagination .= $before . str_replace( 'class="page-numbers"', 'data-barba', $page ) . $after;
				}
			}
			$count++;
		}

		$pagination .= '</div>';
	}

	return $pagination;
}
