<?php
/**
 * Hellix functions.
 *
 * @package hellix
 */

use Elementor\Plugin;

/**
 * Call a shortcode function by tag name.
 *
 * @param string $tag     The shortcode whose function to call.
 * @param array  $atts    The attributes to pass to the shortcode function. Optional.
 * @param array  $content The shortcode's content. Default is null (none).
 *
 * @return string|bool False on failure, the result of the shortcode on success.
 */
function hellix_do_shortcode( $tag, array $atts = array(), $content = null ) {
	global $shortcode_tags;

	if ( ! isset( $shortcode_tags[ $tag ] ) ) {
		return false;
	}

	return call_user_func( $shortcode_tags[ $tag ], $atts, $content, $tag );
}


if ( ! function_exists( 'hellix_pagination' ) ) {
	/**
	 * Display hellix pagination.
	 */
	function hellix_pagination() {
		$max_pages = isset( $GLOBALS['wp_query']->max_num_pages ) ? $GLOBALS['wp_query']->max_num_pages : 1;
		if ( $max_pages < 2 ) {
			return;
		}

		$paged = get_query_var( 'paged' ) ? (int) get_query_var( 'paged' ) : 1;
		$links = paginate_links(
			apply_filters(
				'hellix_posts_pagination_args',
				[
					'type'      => 'array',
					'mid_size'  => 2,
					'prev_next' => false,
				]
			)
		);

		?>
		<div class="pagination -section">
			<?php if ( $paged && 1 < $paged ) : ?>
				<div class="pagination__nav">
					<a data-barba href="<?php echo esc_url( get_previous_posts_page_link() ); ?>" class="nav-icon -left">
						<span class="sr-only"><?php echo esc_html_x( 'Prev', 'front-end', 'hellix' ); ?></span>
						<div class="nav-icon__circle">
							<i class="icon icon-left-arrow"></i>
						</div>
					</a>
				</div>
			<?php endif; ?>
			<div class="pagination__content">
				<?php foreach ( $links as $link ) : ?>
					<?php
					if ( false !== strpos( $link, 'current' ) ) :
						?>
						<span class="is-active">
							<?php echo wp_kses_post( $paged ); ?>
						</span>
					<?php else : ?>
						<?php
						if ( ! ( $max_pages === $paged ) ) {
							?>
							<?php echo wp_kses_post( $link ); ?>
						<?php } else { ?>
							<span>...</span>
							<a data-barba href="<?php echo esc_url( $link ); ?>"><?php echo wp_kses_post( $max_pages ); ?></a>
						<?php } ?>
					<?php endif; ?>
				<?php endforeach; ?>
				</div>

			<?php if ( $paged && $paged < $max_pages ) : ?>
				<div class="pagination__nav">
					<a data-barba href="<?php echo esc_url( get_next_posts_page_link() ); ?>" class="nav-icon -right">
						<span class="sr-only"><?php echo esc_html_x( 'Next', 'front-end', 'hellix' ); ?></span>
						<div class="nav-icon__circle">
							<i class="icon icon-right-arrow"></i>
						</div>
					</a>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'hellix_is_ocdi_activated' ) ) {
	/**
	 * Check if One Click Demo Import is activated
	 */
	function hellix_is_ocdi_activated() {
		return class_exists( 'OCDI_Plugin' ) ? true : false;
	}
}

if ( ! function_exists( 'hellix_is_woocommerce_activated' ) ) {
	/**
	 * Query WooCommerce activation
	 */
	function hellix_is_woocommerce_activated() {
		return class_exists( 'WooCommerce' ) ? true : false;
	}
}

if ( ! function_exists( 'hellix_get_single_portfolio_version' ) ) {
	/**
	 * To get the Single Portfolio Version
	 */
	function hellix_get_single_portfolio_version() {
		$style = hellix_acf_portfolio_type();

		if ( $style ) {
			$portfolio_version = $style;
		} else {
			$portfolio_version = get_theme_mod( 'single_portfolio_version', 'v1' );
		}

		return apply_filters( 'hellix_single_portfolio_version', $portfolio_version );
	}
}


if ( ! function_exists( 'hellix_get_header_version' ) ) {

	/**
	 * To get the Header Version
	 */
	function hellix_get_header_version() {
		$custom_header = hellix_get_field( 'hellix_enable_custom_header' );

		$hx_page_options = array();

		if ( function_exists( 'hellix_option_enabled_post_types' ) && is_singular( hellix_option_enabled_post_types() ) ) {
			$clean_meta_data  = get_post_meta( get_the_ID(), '_hx_page_options', true );
			$_hx_page_options = maybe_unserialize( $clean_meta_data );

			if ( is_array( $_hx_page_options ) ) {
				$hx_page_options = $_hx_page_options;
			}
		}

		if ( isset( $hx_page_options['header'] ) && isset( $hx_page_options['header']['hellix_enable_custom_header'] ) && ( 'yes' === $hx_page_options['header']['hellix_enable_custom_header'] ) ) {
			$version = isset( $hx_page_options['header']['hellix_header_variant'] ) ? $hx_page_options['header']['hellix_header_variant'] : 'v1';
		} elseif ( $custom_header ) {
			$version = hellix_get_field( 'hellix_header_variant' );
		} else {
			$version = get_theme_mod( 'navbar_variant', 'v1' );
		}

		return apply_filters( 'hellix_header_version', $version );
	}
}

if ( ! function_exists( 'hellix_is_mas_static_content_activated' ) ) {
	/**
	 * Query MAS Static Content activation
	 */
	function hellix_is_mas_static_content_activated() {
		return class_exists( 'Mas_Static_Content' ) ? true : false;
	}
}

/**
 * Query ACF activation
 */
function hellix_is_acf_activated() {
	return function_exists( 'get_field' ) ? true : false;
}

if ( ! function_exists( 'hellix_option_enabled_post_types' ) ) {

	/**
	 * Option enabled post types.
	 *
	 * @return array
	 */
	function hellix_option_enabled_post_types() {
		$post_types = [ 'post', 'page', 'jetpack-portfolio' ];
		return apply_filters( 'hellix_option_enabled_post_types', $post_types );
	}
}

/**
 * Get the single post share enable or disable.
 *
 * @return boolean
 */
if ( ! function_exists( 'hellix_single_post_is_share_enabled' ) ) {
	/**
	 * Get the single post share enable or disable.
	 *
	 * @return boolean
	 */
	function hellix_single_post_is_share_enabled() {
		$social_share_enable_disable = get_theme_mod( 'hellix_single_post_social_share', false );

		return (bool) apply_filters( 'hellix_single_post_is_share_enabled', $social_share_enable_disable );
	}
}

if ( ! function_exists( 'hellix_is_extensions_activated' ) ) {

	/**
	 * Function to check if extensions are activated
	 *
	 * @return array
	 */
	function hellix_is_extensions_activated() {
		return class_exists( 'Hellix_Extensions' ) ? true : false;
	}
}

if ( ! function_exists( 'hellix_render_content' ) ) {
	/**
	 * Hellix render content.
	 *
	 * @param array $post_id  post ID.
	 * @param bool  $echo  echo.
	 */
	function hellix_render_content( $post_id, $echo = false ) {
		if ( did_action( 'elementor/loaded' ) ) {
			$content = Plugin::instance()->frontend->get_builder_content_for_display( $post_id );
		} else {
			$content = get_the_content( null, false, $post_id );
			$content = apply_filters( 'the_content', $content );
			$content = str_replace( ']]>', ']]&gt;', $content );
		}

		if ( $echo ) {
			echo wp_kses_post( $content );
		} else {
			return $content;
		}
	}
}

if ( ! function_exists( 'hellix_clean' ) ) {
	/**
	 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
	 * Non-scalar values are ignored.
	 *
	 * @param string|array $var Data to sanitize.
	 * @return string|array
	 */
	function hellix_clean( $var ) {
		if ( is_array( $var ) ) {
			return array_map( 'hellix_clean', $var );
		} else {
			return is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
		}
	}
}

require get_template_directory() . '/inc/functions/template.php';

