<?php
/**
 * Hellix OCDI Class
 *
 * @package mytavel
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Hellix_OCDI' ) ) :
	/**
	 * The one click demo import class.
	 */
	class Hellix_OCDI {

		/**
		 * Stores the assets URL.
		 *
		 * @var string
		 */
		public $assets_url;

		/**
		 * Stores the demo URL.
		 *
		 * @var string
		 */
		public $demo_url;

		/**
		 * Instantiate the class.
		 */
		public function __construct() {

			$this->assets_url = 'https://transvelo.github.io/hellix/assets/';
			$this->demo_url   = 'https://hellix.madrasthemes.com/';

			add_filter( 'pt-ocdi/confirmation_dialog_options', [ $this, 'confirmation_dialog_options' ], 10, 1 );

			add_action( 'pt-ocdi/import_files', [ $this, 'import_files' ] );

			add_action( 'pt-ocdi/after_import', [ $this, 'import_wpforms' ] );
			add_action( 'pt-ocdi/enable_wp_customize_save_hooks', '__return_true' );
			add_filter( 'pt-ocdi/disable_pt_branding', '__return_true' );
			add_action( 'pt-ocdi/after_import', [ $this, 'replace_uploads_dir' ] );

			add_filter( 'ocdi/register_plugins', [ $this, 'register_plugins' ] );
		}

		/**
		 * Register plugins in OCDI.
		 *
		 * @param array $plugins List of plugins.
		 */
		public function register_plugins( $plugins ) {
			global $hellix;

			$profile = 'default';

			if ( isset( $_GET['import'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$demo_id = absint( $_GET['import'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				switch ( $demo_id ) {
					case 7:
						$profile = 'static';
						break;
					case 8:
						$profile = 'portfolio';
						break;
					case 9:
						$profile = 'static';
						break;
					case 13:
						$profile = 'static';
						break;
					case 14:
						$profile = 'portfolio';
						break;
					case 16:
						$profile = 'static';
						break;
					case 17:
						$profile = 'contact';
						break;
				}
			}

			return $hellix->plugin_install->get_demo_plugins( $profile );
		}

		/**
		 * Confirmation dialog box options.
		 *
		 * @param array $options The dialog options.
		 * @return array
		 */
		public function confirmation_dialog_options( $options ) {
			return array_merge(
				$options,
				array(
					'width' => 410,
				)
			);
		}

		/**
		 * Trigger after import
		 */
		public function trigger_ocdi_after_import() {
			$import_files    = $this->import_files();
			$selected_import = end( $import_files );
			do_action( 'pt-ocdi/after_import', $selected_import ); //phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		}

		/**
		 * Replace uploads Dir.
		 *
		 * @param array $selected_import The import that just ran.
		 */
		public function replace_uploads_dir( $selected_import ) {
			if ( isset( $selected_import['uploads_dir'] ) ) {
				$from = $selected_import['uploads_dir'] . '/';
			} else {
				$from = 'https://demo.madrasthemes.com/hellix/';
			}

			$site_upload_dir = wp_get_upload_dir();
			$to              = $site_upload_dir['baseurl'] . '/';
			\Elementor\Utils::replace_urls( $from, $to );
		}

		/**
		 * Clear default widgets.
		 */
		public function clear_default_widgets() {
			$sidebars_widgets = get_option( 'sidebars_widgets' );
			$all_widgets      = array();

			array_walk_recursive(
				$sidebars_widgets,
				function ( $item, $key ) use ( &$all_widgets ) {
					if ( ! isset( $all_widgets[ $key ] ) ) {
						$all_widgets[ $key ] = $item;
					} else {
						$all_widgets[] = $item;
					}
				}
			);

			if ( isset( $all_widgets['array_version'] ) ) {
				$array_version = $all_widgets['array_version'];
				unset( $all_widgets['array_version'] );
			}

			$new_sidebars_widgets = array_fill_keys( array_keys( $sidebars_widgets ), array() );

			$new_sidebars_widgets['wp_inactive_widgets'] = $all_widgets;
			if ( isset( $array_version ) ) {
				$new_sidebars_widgets['array_version'] = $array_version;
			}

			update_option( 'sidebars_widgets', $new_sidebars_widgets );
		}

		/**
		 * Set site options.
		 *
		 * @param array $selected_import The import that just ran.
		 */
		public function set_site_options( $selected_import ) {
			if ( isset( $selected_import['set_pages'] ) && $selected_import['set_pages'] ) {
				$front_page_title = isset( $selected_import['front_page_title'] ) ? $selected_import['front_page_title'] : 'homev1-copy';
				$front_page_id    = get_page_by_title( $front_page_title );

				update_option( 'show_on_front', 'page' );
				update_option( 'page_on_front', $front_page_id->ID );
			}

			update_option( 'posts_per_page', 9 );
		}

		/**
		 * Set the nav menus.
		 *
		 * @param array $selected_import The import that just ran.
		 */
		public function set_nav_menus( $selected_import ) {
			if ( isset( $selected_import['set_nav_menus'] ) && $selected_import['set_nav_menus'] ) {
				$main_menu   = get_term_by( 'name', 'Main Menu', 'nav_menu' );
				$social_menu = get_term_by( 'name', 'Social Icons', 'nav_menu' );
				$locations   = [
					'navbar_nav'   => $main_menu->term_id,
					'social_media' => $social_menu->term_id,
				];

				set_theme_mod( 'nav_menu_locations', $locations );
			}
		}

		/**
		 * Import WPForms.
		 */
		public function import_wpforms() {

			$ocdi_host = 'https://transvelo.github.io/hellix';
			$dd_url    = $ocdi_host . '/assets/wpforms/';

			if ( ! function_exists( 'wpforms' ) || get_option( 'hellix_wpforms_imported', false ) ) {
				return;
			}

			$wpform_file_url = $dd_url . 'all.json';
			$response        = wp_remote_get( $wpform_file_url );

			if ( is_wp_error( $response ) || 200 !== $response['response']['code'] ) {
				return;
			}

			$form_json = wp_remote_retrieve_body( $response );

			if ( is_wp_error( $form_json ) ) {
				return;
			}

			$forms = json_decode( $form_json, true );

			foreach ( $forms as $form_data ) {
				$form_title = $form_data['settings']['form_title'];

				if ( ! empty( $form_data['id'] ) ) {
					$form_content = array(
						'field_id' => '0',
						'settings' => array(
							'form_title' => sanitize_text_field( $form_title ),
							'form_desc'  => '',
						),
					);

					// Merge args and create the form.
					$form = array(
						'import_id'    => (int) $form_data['id'],
						'post_title'   => esc_html( $form_title ),
						'post_status'  => 'publish',
						'post_type'    => 'wpforms',
						'post_content' => wpforms_encode( $form_content ),
					);

					$form_id = wp_insert_post( $form );
				} else {
					// Create initial form to get the form ID.
					$form_id = wpforms()->form->add( $form_title );
				}

				if ( empty( $form_id ) ) {
					continue;
				}

				$form_data['id'] = $form_id;
				// Save the form data to the new form.
				wpforms()->form->update( $form_id, $form_data );
			}

			update_option( 'hellix_wpforms_imported', true );
		}

		/**
		 * Import Files from each demo.
		 */
		public function import_files() {
			$ocdi_host   = 'https://transvelo.github.io/hellix';
			$dd_url      = $ocdi_host . '/assets/xml/';
			$widget_url  = $ocdi_host . '/assets/widgets/';
			$preview_url = $ocdi_host . '/assets/img/ocdi/';
			/* translators: %1$s - The demo name. %2$s - Minutes. */
			$notice  = esc_html__( 'This demo will import %1$s. It may take %2$s', 'hellix' );
			$notice .= '<br><br>' . esc_html__( 'Menus, Widgets and Settings will not be imported. Content imported already will not be imported.', 'hellix' );
			$notice .= '<br><br>' . esc_html__( 'Alternatively, you can import this template directly into your page via Edit with Elementor.', 'hellix' );

			return apply_filters(
				'hellix_ocdi_files_args',
				array(
					array(
						'import_file_name'         => 'Home v1',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v1.xml',
						'import_preview_image_url' => $preview_url . 'home-v1.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/homev1-copy/wp-content/uploads/sites/30',
					),
					array(
						'import_file_name'         => 'Home v2',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v2.xml',
						'import_preview_image_url' => $preview_url . 'home-v2.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v2/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v2/wp-content/uploads/sites/3',
					),
					array(
						'import_file_name'         => 'Home v3',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v3.xml',
						'import_preview_image_url' => $preview_url . 'home-v3.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v3/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v3/wp-content/uploads/sites/9',
					),
					array(
						'import_file_name'         => 'Home v4',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v4.xml',
						'import_preview_image_url' => $preview_url . 'home-v4.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v4/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v4/wp-content/uploads/sites/10',
					),
					array(
						'import_file_name'         => 'Home v5',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v5.xml',
						'import_preview_image_url' => $preview_url . 'home-v5.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v5/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v5/wp-content/uploads/sites/11',
					),
					array(
						'import_file_name'         => 'Home v6',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v6.xml',
						'import_preview_image_url' => $preview_url . 'home-v6.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v6/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v6/wp-content/uploads/sites/14',
					),
					array(
						'import_file_name'         => 'Home v7',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v7.xml',
						'import_preview_image_url' => $preview_url . 'home-v7.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v7/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v7/wp-content/uploads/sites/15',
					),
					array(
						'import_file_name'         => 'Home v8',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v8.xml',
						'import_preview_image_url' => $preview_url . 'home-v8.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v8/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v8/wp-content/uploads/sites/16',
					),
					array(
						'import_file_name'         => 'Home v9',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v9.xml',
						'import_preview_image_url' => $preview_url . 'home-v9.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v9/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v9/wp-content/uploads/sites/17',
					),
					array(
						'import_file_name'         => 'Home v10',
						'categories'               => array( 'Landing Page' ),
						'import_file_url'          => $dd_url . 'home-v10.xml',
						'import_preview_image_url' => $preview_url . 'home-v10.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '25 items including 1 page & 24 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/home-v9/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/home-v10/wp-content/uploads/sites/18',
					),
					array(
						'import_file_name'         => 'About',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'about.xml',
						'import_preview_image_url' => $preview_url . 'about.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '46 items including 2 pages & 44 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/about/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/about/wp-content/uploads/sites/6',
					),
					array(
						'import_file_name'         => 'Contact',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'contact.xml',
						'import_preview_image_url' => $preview_url . 'contact.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '2 items including 2 pages', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/contact/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/contact-us/wp-content/uploads/sites/22',
					),
					array(
						'import_file_name'         => 'Service',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'services.xml',
						'import_preview_image_url' => $preview_url . 'service.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/service/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/services-single/wp-content/uploads/sites/26',
					),
					array(
						'import_file_name'         => 'FAQ',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'faq.xml',
						'import_preview_image_url' => $preview_url . 'faq.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/faqs/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/about/wp-content/uploads/sites/25',
					),
					array(
						'import_file_name'         => 'Team',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'team.xml',
						'import_preview_image_url' => $preview_url . 'team.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/team/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/team/wp-content/uploads/sites/23',
					),
					array(
						'import_file_name'         => 'Pricing',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'pricing.xml',
						'import_preview_image_url' => $preview_url . 'pricing.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/pricing/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/pricing/wp-content/uploads/sites/24',
					),
					array(
						'import_file_name'         => 'Coming Soon',
						'categories'               => array( 'Pages' ),
						'import_file_url'          => $dd_url . 'coming-soon.xml',
						'import_preview_image_url' => $preview_url . 'coming-soon.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/coming-soon/',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/coming-soon/wp-content/uploads/sites/21',
					),
					array(
						'import_file_name'         => 'Blog',
						'categories'               => array( 'Content' ),
						'import_file_url'          => $dd_url . 'blog.xml',
						'import_preview_image_url' => $preview_url . 'blog.png',
						'import_widget_file_url'   => $widget_url . 'widgets.wie',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/blog-list-with-sidebar',
						'plugin_profile'           => 'default',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/blog-lists/wp-content/uploads/sites/32',
					),
					array(
						'import_file_name'         => 'Shop',
						'categories'               => array( 'Shop' ),
						'import_file_url'          => $dd_url . 'shop.xml',
						'import_preview_image_url' => $preview_url . 'shop.png',
						'import_widget_file_url'   => $widget_url . 'widgets.wie',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '29 items including 5 pages, 12 posts, 1 static-content & 11 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/blog-list-with-sidebar',
						'plugin_profile'           => 'shop',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/blog-pages/wp-content/uploads/sites/33',
					),
					array(
						'import_file_name'         => 'Portfolio',
						'categories'               => array( 'Content' ),
						'import_file_url'          => $dd_url . 'portfolio.xml',
						'import_preview_image_url' => $preview_url . 'portfolio.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '46 items including 3 pages, 16 projects & 27 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'preview_url'              => 'https://hellix.madrasthemes.com/portfolio-grid/',
						'plugin_profile'           => 'portfolio',
						'uploads_dir'              => 'https://demo.madrasthemes.com/hellix/shop/wp-content/uploads/sites/13',
					),
					array(
						'import_file_name'         => 'Static Content',
						'categories'               => array( 'Others' ),
						'import_file_url'          => $dd_url . 'static-content.xml',
						'import_preview_image_url' => $preview_url . 'footer.png',
						'import_notice'            => sprintf( $notice, '<strong>' . esc_html__( '16 items including 11 Static Contents & 5 images', 'hellix' ) . '</strong>', esc_html__( 'upto a minute', 'hellix' ) ),
						'plugin_profile'           => 'default',

					),
					array(
						'import_file_name'         => 'Menus',
						'categories'               => array( 'Others' ),
						'import_file_url'          => $dd_url . 'menu.xml',
						'import_preview_image_url' => $preview_url . 'menu.png',
						'plugin_profile'           => 'static',
					),
				)
			);
		}
	}

endif;

return new Hellix_OCDI();
