<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Quarty Blog Carousel Widget.
 *
 * @since 1.0
 */
class Quarty_Blog_Carousel_Widget extends Widget_Base {

	public function get_name() {
		return 'quarty-blog-carousel';
	}

	public function get_title() {
		return esc_html__( 'Blog (Carousel)', 'quarty-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'quarty-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'heading_tab',
			[
				'label' => esc_html__( 'Title', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'     => esc_html__( 'Title', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'quarty-plugin' ),
					'h2' => __( 'H2', 'quarty-plugin' ),
					'h3' => __( 'H3', 'quarty-plugin' ),
					'h4' => __( 'H4', 'quarty-plugin' ),
					'div' => __( 'DIV', 'quarty-plugin' ),
				],
			]
		);

		$this->add_control(
			'title_show',
			[
				'label' => esc_html__( 'Show Title', 'quarty-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', 'quarty-plugin' ),
				'label_off' => __( 'Hide', 'quarty-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'source',
			[
				'label'       => esc_html__( 'Source', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'all',
				'options' => [
					'all'  => __( 'All', 'quarty-plugin' ),
					'categories' => __( 'Categories', 'quarty-plugin' ),
				],
			]
		);

		$this->add_control(
			'source_categories',
			[
				'label'       => esc_html__( 'Source', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple' => true,
				'options' => $this->get_blog_categories(),
				'condition' => [
		            'source' => 'categories'
		        ],
			]
		);

		$this->add_control(
			'limit',
			[
				'label'       => esc_html__( 'Number of Items', 'quarty-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => 6,
				'default'     => 6,
			]
		);

		$this->add_control(
			'sort',
			[
				'label'       => esc_html__( 'Sorting By', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'date'  => __( 'Date', 'quarty-plugin' ),
					'title' => __( 'Title', 'quarty-plugin' ),
					'rand' => __( 'Random', 'quarty-plugin' ),
					'menu_order' => __( 'Order', 'quarty-plugin' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'       => esc_html__( 'Order', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => __( 'ASC', 'quarty-plugin' ),
					'desc' => __( 'DESC', 'quarty-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-section .qrt-title-h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .qrt-section .qrt-title-h',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_details_color',
			[
				'label'     => esc_html__( 'Details Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-post-frame .qrt-post-thumb .qrt-post-date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'item_details_bg',
			[
				'label'     => esc_html__( 'Details Background', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-post-frame .qrt-post-thumb .qrt-post-date' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_details_typography',
				'label'     => esc_html__( 'Details Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-post-frame .qrt-post-thumb .qrt-post-date',
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-post-frame .qrt-post-descr .qrt-post-title, {{WRAPPER}} .qrt-post-frame .qrt-post-descr .qrt-post-title a' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-post-frame .qrt-post-descr .qrt-post-title',
			]
		);

		$this->add_control(
			'item_text_color',
			[
				'label'     => esc_html__( 'Text Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-post-frame .qrt-post-descr .qrt-port-short-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_text_typography',
				'label'     => esc_html__( 'Text Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-post-frame .qrt-post-descr .qrt-port-short-text',
			]
		);
		
		$this->end_controls_section();
	}

	/**
	 * Render Categories List.
	 *
	 * @since 1.0
	 */
	protected function get_blog_categories() {
		$categories = [];

		$args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'category',
			'pad_counts'	=> false 
		);

		$blog_categories = get_categories( $args );

		foreach ( $blog_categories as $category ) {
			$categories[$category->term_id] = $category->name;
		}

		return $categories;
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$this->add_inline_editing_attributes( 'title', 'basic' );

		if ( $settings['source'] == 'all' ) {
			$cat_ids = '';
		} else {
			$cat_ids = $settings['source_categories'];
		}

		$cat_args = array(
			'type'			=> 'post',
			'child_of'		=> 0,
			'parent'		=> '',
			'orderby'		=> 'name',
			'order'			=> 'DESC',
			'hide_empty'	=> 1,
			'hierarchical'	=> 1,
			'taxonomy'		=> 'category',
			'pad_counts'	=> false,
			'include'		=> $cat_ids
		);

		$bp_categories = get_categories( $cat_args );

		$args = array(
			'post_type'			=> 'post',
			'post_status'		=> 'publish',
			'orderby'			=> $settings['sort'],
			'order'				=> $settings['order'],
			'posts_per_page'	=> $settings['limit'],
			'paged' 			=> 1
		);

		if( $settings['source'] == 'categories' ) {
			$tax_array = array(
				array(
					'taxonomy' => 'category',
					'field'    => 'id',
					'terms'    => $cat_ids
				)
			);

			$args += array('tax_query' => $tax_array);
		}

		$q = new \WP_Query( $args );

		?>

		<div class="qrt-section">	
			<div class="row">
			  <?php if ( $settings['title'] && $settings['title_show'] == 'yes' ) : ?>
              <div class="col-lg-12">
                <<?php echo esc_attr( $settings['title_tag'] ); ?> class="qrt-mb-40 qrt-title-h">
                	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          		<?php echo wp_kses_post( $settings['title'] ); ?>
		         	</span>
            	</<?php echo esc_attr( $settings['title_tag'] ); ?>>
              </div>
              <?php endif; ?>

              <?php if ( $q->have_posts() ) : ?>
              <div class="col-lg-12">
                <div class="swiper-container qrt-pop-post-slider">
                  <div class="swiper-wrapper">
                    <?php while ( $q->have_posts() ) : $q->the_post(); ?>
                    <div class="swiper-slide">
                    	<?php get_template_part( 'template-parts/content-carousel' ); ?>
                    </div>
                    <?php endwhile; ?>
                  </div>
                </div>
              </div>
              <div class="col-6">
                <div class="swiper-pp-pagination qrt-cursor-color qrt-cursor-scale"></div>
              </div>
              <div class="col-6">
                <div class="qrt-slider-nav">
                  <!-- prev -->
                  <div class="qrt-slider-prev qrt-pp-prev qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-left"></i></div>
                  <!-- next -->
                  <div class="qrt-slider-next qrt-pp-next qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-right"></i></div>
                </div>
              </div>
            </div>
            <?php endif; wp_reset_postdata(); ?>
        </div>

		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Quarty_Blog_Carousel_Widget() );