<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Quarty Services Widget.
 *
 * @since 1.0
 */
class Quarty_Services_Widget extends Widget_Base {

	public function get_name() {
		return 'quarty-services';
	}

	public function get_title() {
		return esc_html__( 'Services', 'quarty-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'quarty-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'     => esc_html__( 'Title', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'quarty-plugin' ),
					'h2' => __( 'H2', 'quarty-plugin' ),
					'h3' => __( 'H3', 'quarty-plugin' ),
					'h4' => __( 'H4', 'quarty-plugin' ),
					'div' => __( 'DIV', 'quarty-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Icon', 'quarty-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'name', [
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter title', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'desc', [
				'label'       => esc_html__( 'Description', 'quarty-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter description', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter description', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Services Items', 'quarty-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-section .qrt-title-h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .qrt-section .qrt-title-h',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_title_color',
			[
				'label'     => esc_html__( 'Title Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-service-item .qrt-service-title' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_title_typography',
				'label'     => esc_html__( 'Title Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-service-item .qrt-service-title',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-service-item .qrt-service-desc' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-service-item .qrt-service-desc',
			]
		);
		
		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		
		$this->add_inline_editing_attributes( 'title', 'basic' );

		?>

		<div class="qrt-section qrt-section-sf">
			<div class="row">
			  <?php if ( $settings['title'] ) : ?>
              <div class="col-lg-12">
                <<?php echo esc_attr( $settings['title_tag'] ); ?> class="qrt-mb-40 qrt-title-h">
                	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          		<?php echo wp_kses_post( $settings['title'] ); ?>
		         	</span>
                </<?php echo esc_attr( $settings['title_tag'] ); ?>>
              </div>
              <?php endif; ?>

              <?php if ( $settings['items'] ) : ?>
              <?php foreach ( $settings['items'] as $index => $item ) : 
			  $item_name = $this->get_repeater_setting_key( 'name', 'items', $index );
			  $this->add_inline_editing_attributes( $item_name, 'basic' );

			  $item_desc = $this->get_repeater_setting_key( 'desc', 'items', $index );
			  $this->add_inline_editing_attributes( $item_desc, 'advanced' );
			  ?>
              <div class="col-lg-4 qrt-service-item">
                <div class="qrt-icon-box">
                  <?php if ( $item['image'] ) : $icon = wp_get_attachment_image_url( $item['image']['id'], 'quarty_256x256' ); ?>
                  <img class="mb-20" src="<?php echo esc_url( $icon ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>">
                  <?php endif; ?>

                  <?php if ( $item['name'] ) : ?>
                  <h4 class="mb-20 qrt-service-title">
                  	<span <?php echo $this->get_render_attribute_string( $item_name ); ?>>
		        		<?php echo wp_kses_post( $item['name'] ); ?>
		        	</span>
                  </h4>
                  <?php endif; ?>

                  <?php if ( $item['desc'] ) : ?>
                  <div class="qrt-service-desc">	
                  	<div <?php echo $this->get_render_attribute_string( $item_desc ); ?>>
		        		<?php echo wp_kses_post( $item['desc'] ); ?>
		          	</div>
		          </div>
                  <?php endif; ?>
                </div>
              </div>
              <?php endforeach; ?>
              <?php endif; ?>
            </div>
        </div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>

		<div class="qrt-section qrt-section-sf">
			<div class="row">
			  <# if ( settings.title ) { #>
              <div class="col-lg-12">
                <{{{ settings.title_tag }}} class="qrt-mb-40 qrt-title-h">
                	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          		{{{ settings.title }}}
		         	</span>
                </{{{ settings.title_tag }}}>
              </div>
              <# } #>

              <# if ( settings.items ) { #>
              <# _.each( settings.items, function( item, index ) { 

				var item_name = view.getRepeaterSettingKey( 'name', 'items', index );
				view.addInlineEditingAttributes( item_name, 'basic' );

				var item_desc = view.getRepeaterSettingKey( 'desc', 'items', index );
				view.addInlineEditingAttributes( item_desc, 'advanced' );

			  #>
              <div class="col-lg-4 qrt-service-item">
                <div class="qrt-icon-box">
                  <# if ( item.image ) { #>
                  <img class="mb-20" src="{{{ item.image.url }}}" alt="{{{ item.name }}}">
                  <# } #>

                  <# if ( item.name ) { #>
                  <h4 class="mb-20 qrt-service-title">
                  	<span {{{ view.getRenderAttributeString( item_name ) }}}>
						{{{ item.name }}}
					</span>
                  </h4>
                  <# } #>

                  <# if ( item.desc ) { #>
                  <div class="qrt-service-desc">	
                  	<div {{{ view.getRenderAttributeString( item_desc ) }}}>
						{{{ item.desc }}}
					</div>
		          </div>
                  <# } #>
                </div>
              </div>
              <# }); #>
              <# } #>
            </div>
        </div>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Quarty_Services_Widget() );