<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Quarty Team Widget.
 *
 * @since 1.0
 */
class Quarty_Team_Widget extends Widget_Base {

	public function get_name() {
		return 'quarty-team';
	}

	public function get_title() {
		return esc_html__( 'Team', 'quarty-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'quarty-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'     => esc_html__( 'Title', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'quarty-plugin' ),
					'h2' => __( 'H2', 'quarty-plugin' ),
					'h3' => __( 'H3', 'quarty-plugin' ),
					'h4' => __( 'H4', 'quarty-plugin' ),
					'div' => __( 'DIV', 'quarty-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'type' => Controls_Manager::TEXT,
				'label' => esc_html__( 'Image', 'quarty-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'name', [
				'label'       => esc_html__( 'Name', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter name', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter name', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'subname', [
				'label'       => esc_html__( 'Subname', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subname', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter subname', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Team Items', 'quarty-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-section .qrt-title-h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .qrt-section .qrt-title-h',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_name_color',
			[
				'label'     => esc_html__( 'Name Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-member-name h4' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_name_typography',
				'label'     => esc_html__( 'Name Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-member-name h4',
			]
		);

		$this->add_control(
			'item_subname_color',
			[
				'label'     => esc_html__( 'Subname Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-member-name > span' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_subname_typography',
				'label'     => esc_html__( 'Subname Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-member-name > span',
			]
		);

		$this->add_control(
			'item_bg_color',
			[
				'label'     => esc_html__( 'Description Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-team-member .qrt-team-member-description' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		
		?>

		<div class="qrt-section qrt-section-sf">
			<div class="row">
              <?php if ( $settings['title'] ) : ?>
              <div class="col-lg-12">
                <<?php echo esc_attr( $settings['title_tag'] ); ?> class="qrt-title-h qrt-mb-40">
            		<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          		<?php echo wp_kses_post( $settings['title'] ); ?>
		         	</span>
            	</<?php echo esc_attr( $settings['title_tag'] ); ?>>
              </div>
              <?php endif; ?>

              <?php if ( $settings['items'] ) : ?>
              <?php foreach ( $settings['items'] as $index => $item ) : 
			    $item_name = $this->get_repeater_setting_key( 'name', 'items', $index );
			    $this->add_inline_editing_attributes( $item_name, 'none' );

			    $item_subname = $this->get_repeater_setting_key( 'subname', 'items', $index );
			    $this->add_inline_editing_attributes( $item_subname, 'none' );
			  ?>
              <div class="<?php if ( get_field( 'page_layout' ) != 1 ) : ?>col-lg-3<?php else : ?>col-lg-4<?php endif; ?>">
                <div class="qrt-team-member">
                  <?php if ( $item['image'] ) : $image = wp_get_attachment_image_url( $item['image']['id'], 'quarty_950xAuto' ); ?>
                  <div class="qrt-cursor-scale qrt-team-item qrt-anima-link">
                    <img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>" />
                  </div>
                  <?php endif; ?>
                  <?php if ( $item['name'] || $item['subname'] ) : ?>
                  <div class="qrt-team-member-description">
                    <div class="qrt-member-name">
                      <h4 class="qrt-white qrt-qrt-mb-5">
                      	<span <?php echo $this->get_render_attribute_string( $item_name ); ?>>
							<?php echo wp_kses_post( $item['name'] ); ?>
						</span>
                      </h4>
                      <span <?php echo $this->get_render_attribute_string( $item_subname ); ?>>
						<?php echo wp_kses_post( $item['subname'] ); ?>
					  </span>
                    </div>
                  </div>
                  <?php endif; ?>
                </div>
              </div>
              <?php endforeach; ?>
              <?php endif; ?>
            </div>
        </div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>

		<div class="qrt-section qrt-section-sf">
			<div class="row">
              <# if ( settings.title ) { #>
              <div class="col-lg-12">
                <{{{ settings.title_tag }}} class="qrt-title-h qrt-mb-40">
            		<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          		{{{ settings.title }}}
		        	</span>
            	</{{{ settings.title_tag }}}>
              </div>
              <# } #>

              <# if ( settings.items ) { #>
              <# _.each( settings.items, function( item, index ) {
			    var item_name = view.getRepeaterSettingKey( 'name', 'items', index );
			    view.addInlineEditingAttributes( item_name, 'none' );

			    var item_subname = view.getRepeaterSettingKey( 'subname', 'items', index );
			    view.addInlineEditingAttributes( item_subname, 'none' );
			  #>
              <div class="col-lg-3">
                <div class="qrt-team-member">
                  <# if ( item.image ) { #>
                  <div class="qrt-cursor-scale qrt-team-item qrt-anima-link">
                    <img src="{{{ item.image.url }}}" alt="{{{ item.name }}}" />
                  </div>
                  <# } #>
                  <# if ( item.name || item.subname ) { #>
                  <div class="qrt-team-member-description">
                    <div class="qrt-member-name">
                      <h4 class="qrt-white qrt-qrt-mb-5">
                      	<span {{{ view.getRenderAttributeString( item_name ) }}}>
							{{{ item.name }}}
						</span>
                      </h4>
                      <span {{{ view.getRenderAttributeString( item_subname ) }}}>
						{{{ item.subname }}}
					  </span>
                    </div>
                  </div>
                  <# } #>
                </div>
              </div>
              <# }); #>
              <# } #>
            </div>
        </div>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Quarty_Team_Widget() );