<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Quarty Testimonials Widget.
 *
 * @since 1.0
 */
class Quarty_Testimonials_Widget extends Widget_Base {

	public function get_name() {
		return 'quarty-testimonials';
	}

	public function get_title() {
		return esc_html__( 'Testimonials', 'quarty-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'quarty-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'     => esc_html__( 'Title', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h4',
				'options' => [
					'h1'  => __( 'H1', 'quarty-plugin' ),
					'h2' => __( 'H2', 'quarty-plugin' ),
					'h3' => __( 'H3', 'quarty-plugin' ),
					'h4' => __( 'H4', 'quarty-plugin' ),
					'div' => __( 'DIV', 'quarty-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Image', 'quarty-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'name', [
				'label'       => esc_html__( 'Name', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter name', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter name', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'subname', [
				'label'       => esc_html__( 'Subname', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter subname', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter subname', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'desc', [
				'label'       => esc_html__( 'Description', 'quarty-plugin' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter description', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter description', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'rating', [
				'label'       => esc_html__( 'Rating', 'quarty-plugin' ),
				'type'        => Controls_Manager::NUMBER,
				'placeholder' => esc_html__( 'Enter rating', 'quarty-plugin' ),
				'default'	=> 5,
				'min' => 0,
				'max' => 5,
				'step' => 1,
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Testimonials Items', 'quarty-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-section .qrt-title-h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .qrt-section .qrt-title-h',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_styling',
			[
				'label'     => esc_html__( 'Items', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_name_color',
			[
				'label'     => esc_html__( 'Name Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-testimonial .qrt-testimonial-header h4' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_name_typography',
				'label'     => esc_html__( 'Name Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-testimonial .qrt-testimonial-header h4',
			]
		);

		$this->add_control(
			'item_subname_color',
			[
				'label'     => esc_html__( 'Subname Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-testimonial .qrt-testimonial-header .qrt-el-suptitle' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_subname_typography',
				'label'     => esc_html__( 'Subname Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-testimonial .qrt-testimonial-header .qrt-el-suptitle',
			]
		);

		$this->add_control(
			'item_desc_color',
			[
				'label'     => esc_html__( 'Description Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-testimonial .qrt-testimonial-text' => 'color: {{VALUE}};',
				],
			]
		);		

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'item_desc_typography',
				'label'     => esc_html__( 'Description Typography', 'quarty-plugin' ),
				'selector' => '{{WRAPPER}} .qrt-testimonial .qrt-testimonial-text',
			]
		);
		
		$this->add_control(
			'item_rating_color',
			[
				'label'     => esc_html__( 'Rating Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-testimonial .qrt-stars li i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'title', 'basic' );
		
		?>

		<div class="qrt-section">	
			<div class="row">
              <?php if ( $settings['title'] ) : ?>
              <div class="col-lg-12">
                <<?php echo esc_attr( $settings['title_tag'] ); ?> class="qrt-title-h qrt-mb-40">
                	<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          		<?php echo wp_kses_post( $settings['title'] ); ?>
		         	</span>
                </<?php echo esc_attr( $settings['title_tag'] ); ?>>
              </div>
              <?php endif; ?>

              <?php if ( $settings['items'] ) : ?>
              <div class="col-lg-12">
                <div class="swiper-container qrt-testimonials-slider" style="overflow: visible">
                  <div class="swiper-wrapper">
                  	<?php foreach ( $settings['items'] as $index => $item ) : 
				    $item_name = $this->get_repeater_setting_key( 'name', 'items', $index );
				    $this->add_inline_editing_attributes( $item_name, 'none' );

				    $item_subname = $this->get_repeater_setting_key( 'subname', 'items', $index );
				    $this->add_inline_editing_attributes( $item_subname, 'none' );

				    $item_desc = $this->get_repeater_setting_key( 'desc', 'items', $index );
				    $this->add_inline_editing_attributes( $item_desc, 'advanced' );
				    ?>
                    <div class="swiper-slide">
                      <div class="qrt-testimonial">
                        <div class="qrt-testimonial-header">
                          <?php if ( $item['image'] ) : $image = wp_get_attachment_image_url( $item['image']['id'], 'quarty_140x140' ); ?>
                          <img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>">
                          <?php endif; ?>
                          <?php if ( $item['name'] || $item['subname'] ) : ?>
                          <div class="qrt-testimonial-name">
                            <?php if ( $item['name'] ) : ?>
                            <h4 class="qrt-mb-5">
                            	<span <?php echo $this->get_render_attribute_string( $item_name ); ?>>
									<?php echo wp_kses_post( $item['name'] ); ?>
								</span>
                            </h4>
                            <?php endif; ?>
                            <?php if ( $item['subname'] ) : ?>
                            <div class="qrt-el-suptitle">
                            	<span <?php echo $this->get_render_attribute_string( $item_subname ); ?>>
									<?php echo wp_kses_post( $item['subname'] ); ?>
								</span>
                            </div>
                            <?php endif; ?>
                          </div>
                          <?php endif; ?>
                        </div>
                        <?php if ( $item['desc'] ) : ?>
                        <div class="qrt-testimonial-text">
                          <div <?php echo $this->get_render_attribute_string( $item_desc ); ?>>
							<?php echo wp_kses_post( $item['desc'] ); ?>
						  </div>
                        </div>
                        <?php endif; ?>
                        <?php if ( $item['rating'] ) : ?>
                        <ul class="qrt-stars">
                          <?php for ( $i = 1; $i <= 5; $i++ ) : ?>
		                  <?php if ( $item['rating'] >= $i ) : ?>
		                  <li><i class="fas fa-star"></i></li>
		                  <?php else : ?>
		                  <li class="qrt-empty-item"><i class="fas fa-star"></i></li>
		              	  <?php endif; ?>
		                  <?php endfor; ?>
                        </ul>
                        <?php endif; ?>
                      </div>
                    </div>
                    <?php endforeach; ?>
                  </div>
                </div>

              </div>
              <div class="col-6">
                <div class="swiper-testi-pagination qrt-cursor-color qrt-cursor-scale"></div>
              </div>
              <div class="col-6">
                <div class="qrt-slider-nav">
                  <!-- prev -->
                  <div class="qrt-slider-prev qrt-testi-prev qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-left"></i></div>
                  <!-- next -->
                  <div class="qrt-slider-next qrt-testi-next qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-right"></i></div>
                </div>
              </div>
              <?php endif; ?>

            </div>
        </div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>
		
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>

		<div class="qrt-section">	
			<div class="row">
              <# if ( settings.title ) { #>
              <div class="col-lg-12">
                <{{{ settings.title_tag }}} class="qrt-title-h qrt-mb-40">
                	<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          		{{{ settings.title }}}
		        	</span>
                </{{{ settings.title_tag }}}>
              </div>
              <# } #>

              <# if ( settings.items ) { #>
              <div class="col-lg-12">
                <div class="swiper-container qrt-testimonials-slider" style="overflow: visible">
                  <div class="swiper-wrapper">
                  	<# _.each( settings.items, function( item, index ) {
				    var item_name = view.getRepeaterSettingKey( 'name', 'items', index );
				    view.addInlineEditingAttributes( item_name, 'none' );

			    	var item_subname = view.getRepeaterSettingKey( 'subname', 'items', index );
			    	view.addInlineEditingAttributes( item_subname, 'none' );

			    	var item_desc = view.getRepeaterSettingKey( 'desc', 'items', index );
			    	view.addInlineEditingAttributes( item_desc, 'advanced' );
			    	#>
                    <div class="swiper-slide">
                      <div class="qrt-testimonial">
                        <div class="qrt-testimonial-header">
                          <# if ( item.image ) { #>
                          <img src="{{{ item.image.url }}}" alt="{{{ item.name }}}">
                          <# } #>
                          <# if ( item.name || item.subname ) { #>
                          <div class="qrt-testimonial-name">
                            <# if ( item.name ) { #>
                            <h4 class="qrt-mb-5">
                            	<span {{{ view.getRenderAttributeString( item_name ) }}}>
									{{{ item.name }}}
								</span>
                            </h4>
                            <# } #>
                            <# if ( item.subname ) { #>
                            <div class="qrt-el-suptitle">
                            	<span {{{ view.getRenderAttributeString( item_subname ) }}}>
									{{{ item.subname }}}
								</span>
                            </div>
                            <# } #>
                          </div>
                          <# } #>
                        </div>
                        <# if ( item.desc ) { #>
                        <div class="qrt-testimonial-text">
                          <div {{{ view.getRenderAttributeString( item_desc ) }}}>
							{{{ item.desc }}}
						  </div>
                        </div>
                        <# } #>
                        <# if ( item.rating ) { #>
                        <ul class="qrt-stars">
                          <# for ( var i = 0; i < 5; i++ ) { #>
		                  <# if ( item.rating >= i ) { #>
		                  <li><i class="fas fa-star"></i></li>
		                  <# } else { #>
		                  <li class="qrt-empty-item"><i class="fas fa-star"></i></li>
		                  <# } #>
		                  <# } #>
                        </ul>
                        <# } #>
                      </div>
                    </div>
                    <# }); #>
                  </div>
                </div>

              </div>
              <div class="col-6">
                <div class="swiper-testi-pagination qrt-cursor-color qrt-cursor-scale"></div>
              </div>
              <div class="col-6">
                <div class="qrt-slider-nav">
                  <!-- prev -->
                  <div class="qrt-slider-prev qrt-testi-prev qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-left"></i></div>
                  <!-- next -->
                  <div class="qrt-slider-next qrt-testi-next qrt-cursor-color qrt-cursor-scale"><i class="fas fa-arrow-right"></i></div>
                </div>
              </div>
              <# } #>

            </div>
        </div>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Quarty_Testimonials_Widget() );