<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
 
/**
 * Quarty Project Image Gallery Widget.
 *
 * @since 1.0
 */
class Quarty_Project_Image_Gallery_Widget extends Widget_Base {

	public function get_name() {
		return 'quarty-project-image-gallery';
	}

	public function get_title() {
		return esc_html__( 'Project Image Gallery', 'quarty-plugin' );
	}

	public function get_icon() {
		return 'eicon-parallax';
	}

	public function get_categories() {
		return [ 'quarty-category' ];
	}

	/**
	 * Register widget controls.
	 *
	 * @since 1.0
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'title_tab',
			[
				'label' => esc_html__( 'Title', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter title', 'quarty-plugin' ),
				'default'     => esc_html__( 'Title', 'quarty-plugin' ),
			]
		);

		$this->add_control(
			'title_tag',
			[
				'label'       => esc_html__( 'Title Tag', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => [
					'h1'  => __( 'H1', 'quarty-plugin' ),
					'h2' => __( 'H2', 'quarty-plugin' ),
					'h3' => __( 'H3', 'quarty-plugin' ),
					'h4' => __( 'H4', 'quarty-plugin' ),
					'div' => __( 'DIV', 'quarty-plugin' ),
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'items_tab',
			[
				'label' => esc_html__( 'Items', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image', [
				'label' => esc_html__( 'Image', 'quarty-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'name', [
				'label'       => esc_html__( 'Name', 'quarty-plugin' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter name', 'quarty-plugin' ),
				'default'	=> esc_html__( 'Enter name', 'quarty-plugin' ),
			]
		);

		$repeater->add_control(
			'size', [
				'label'       => esc_html__( 'Size', 'quarty-plugin' ),
				'type'        => Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default'  => __( 'Default', 'quarty-plugin' ),
					'vertical' => __( 'Vertical', 'quarty-plugin' ),
					'wide' => __( 'Wide', 'quarty-plugin' ),
				],
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Images', 'quarty-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'prevent_empty' => false,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ name }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'settings_tab',
			[
				'label' => esc_html__( 'Settings', 'quarty-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'masonry',
			[
				'label' => esc_html__( 'Masonry Grid', 'quarty-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Enable', 'quarty-plugin' ),
				'label_off' => __( 'Disable', 'quarty-plugin' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title_styling',
			[
				'label'     => esc_html__( 'Title', 'quarty-plugin' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Color', 'quarty-plugin' ),
				'type'      => Controls_Manager::COLOR,
				'default'	=> '',
				'selectors' => [
					'{{WRAPPER}} .qrt-section .qrt-title-h' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .qrt-section .qrt-title-h',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$this->add_inline_editing_attributes( 'title', 'basic' );

		$theme_lightbox = get_field( 'portfolio_lightbox_disable', 'option' );
		
		?>

		<div class="qrt-section qrt-section-wb">

		<?php if ( $settings['title'] ) : ?>
		<div class="row">
			<div class="col-lg-12">
				<<?php echo esc_attr( $settings['title_tag'] ); ?> class="qrt-title-h">
					<span <?php echo $this->get_render_attribute_string( 'title' ); ?>>
		          		<?php echo wp_kses_post( $settings['title'] ); ?>
		         	</span>
				</<?php echo esc_attr( $settings['title_tag'] ); ?>>
			</div>
		</div>
		<?php endif; ?>

		<?php if ( $settings['items'] ) : ?>
		<div class="qrt-masonry-grid<?php if ( $settings['masonry'] == 'yes' ) : ?> qrt-grid-masonry<?php endif; ?> qrt-masonry-grid-single">
          <div class="qrt-grid-sizer"></div>
          <?php foreach ( $settings['items'] as $index => $item ) : ?>
          <?php if ( $item['image'] ) :  $image = wp_get_attachment_image_url( $item['image']['id'], 'quarty_950xAuto' ); $image_full = wp_get_attachment_image_url( $item['image']['id'], 'quarty_1920xAuto' ); ?>
          <div class="qrt-masonry-grid-item qrt-masonry-grid-item-50<?php if ( $item['size'] == 'vertical' ) : ?> qrt-masonry-grid-item-h-x-2<?php endif; ?><?php if ( $item['size'] == 'wide' ) : ?> qrt-masonry-grid-item-100<?php endif; ?>">
            <div class="qrt-work-item">
            	<a<?php if ( ! $theme_lightbox ) : ?> data-magnific-gallery<?php endif; ?> data-elementor-lightbox-slideshow="gallery" data-no-swup href="<?php echo esc_url( $image_full ); ?>" class="qrt-cursor-scale qrt-work-cover-frame">
            		<img src="<?php echo esc_url( $image ); ?>" alt="<?php echo esc_attr( $item['name'] ); ?>">
                	<div class="qrt-item-zoom qrt-cursor-color"><i class="fas fa-expand"></i></div>
              	</a>
            </div>
          </div>
          <?php endif; ?>
          <?php endforeach; ?>
        </div>
        <?php endif; ?>

    	</div>

		<?php
	}

	/**
	 * Render widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {
		?>
		<#
		view.addInlineEditingAttributes( 'title', 'basic' );
		#>

		<div class="qrt-section qrt-section-wb">

		<# if ( settings.title ) { #>
		<div class="row">
			<div class="col-lg-12">
				<{{{ settings.title_tag }}} class="qrt-title-h">
					<span {{{ view.getRenderAttributeString( 'title' ) }}}>
		          		{{{ settings.title }}}
		         	</span>
				</{{{ settings.title_tag }}}>
			</div>
		</div>
		<# } #>

		<# if ( settings.items ) { #>
		<div class="qrt-masonry-grid<# if ( settings.masonry == 'yes' ) { #> qrt-grid-masonry<# } #>">
          <div class="qrt-grid-sizer"></div>
          <# _.each( settings.items, function( item, index ) { #>
          <# if ( item.image ) { #>
          <div class="qrt-masonry-grid-item qrt-masonry-grid-item-50<# if ( item.size == 'vertical' ) { #> qrt-masonry-grid-item-h-x-2<# } #><# if ( item.size == 'wide' ) { #> qrt-masonry-grid-item-100<# } #>">
            <div class="qrt-work-item">
            	<a data-magnific-gallery data-elementor-lightbox-slideshow="gallery" data-no-swup href="{{{ item.image.url }}}" class="qrt-cursor-scale qrt-work-cover-frame">
            		<img src="{{{ item.image.url }}}" alt="{{{ item.name }}}">
                	<div class="qrt-item-zoom qrt-cursor-color"><i class="fas fa-expand"></i></div>
              	</a>
            </div>
          </div>
          <# } #>
          <# }); #>
        </div>
        <# } #>

    	</div>

		<?php 
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Quarty_Project_Image_Gallery_Widget() );