<?php
/**
 * Royarch Elementor Global Widget Contact_Block.
 *
 * @package    Royarch - Unikwp
 * @subpackage royarch
 * @since      Royarch 1.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	return; // Exit if it is accessed directly
}

class Royarch_Elementor_Global_Widgets_Contact_Block extends Widget_Base {

	/**
	 * Retrieve Royarch_Elementor_Global_Widgets_Contact_Block widget name.
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'Royarch-Global-Widgets-Contact-Block';
	}

	/**
	 * Retrieve Royarch_Elementor_Global_Widgets_Contact_Block widget Title.
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Contact Form Block', 'royarch' );
	}

	/**
	 * Retrieve royarch_Elementor_Global_Widget_Contact_Block widget icon.
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'far fa-envelope';
	}

	/**
	 * Retrieve the list of categories the royarch_Elementor_Global_Widget_Contact_Block widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'royarch-widget-blocks' );
	}
	
	/**
	 * Register royarch_Elementor_Global_Widget_Contact_Block widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @access protected
	 */
	protected function _register_controls() {

		// Widget title section
		$this->start_controls_section(
			'section_Contact_Block_title_manage',
			array(
				'label' => esc_html__( 'Contact Form Block', 'royarch' ),
			)
		);

		$this->add_control(
			'widget_title',
			array(
				'label'       => esc_html__( 'Section Title:', 'royarch' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Add your custom block title', 'royarch' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'widget_sub',
			array(
				'label'       => esc_html__( 'Section SubTitle:', 'royarch' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Add your custom block subtitle', 'royarch' ),
				'label_block' => true,
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'list_title', [
				'label' => __( 'Contact Name', 'royarch' ),
				'type' => Controls_Manager::TEXT,
			]
		);

		$repeater->add_control(
			'list_desc', [
				'label' => __( 'Contact Description', 'royarch' ),
				'type' => Controls_Manager::TEXTAREA,
			]
		);

		$repeater->add_control(
			'list_icon', [
				'label' => __( 'Icon', 'royarch' ),
				'type' => Controls_Manager::ICONS,
			]
		);

		$this->add_control(
			'list',
			[
				'label' => __( 'Contact Info List', 'royarch' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ list_title }}}',
			]
		);

		$this->add_control(
			'widget_contact_title',
			array(
				'label'       => esc_html__( 'Contact Form Title:', 'royarch' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Add your contact form title', 'royarch' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'widget_contact',
			array(
				'label'       => esc_html__( 'Contact Form 7', 'royarch' ),
				'type'        => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Add your Contact Form 7 Shortcode', 'royarch' ),
				'label_block' => true,
			)
		);
        
		$this->end_controls_section();
		
		$this->start_controls_section(
			'section_map',
			[
				'label' => __( 'Map', 'royarch' ),
			]
		);

		$default_address = __( 'London Eye, London, United Kingdom', 'royarch' );
		$this->add_control(
			'address',
			[
				'label' => __( 'Address', 'royarch' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => $default_address,
				'default' => $default_address,
				'label_block' => true,
			]
		);

		$this->add_control(
			'zoom',
			[
				'label' => __( 'Zoom Level', 'royarch' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 10,
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
			]
		);

		$this->add_control(
			'height',
			[
				'label' => __( 'Height', 'royarch' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 300,
				],
				'range' => [
					'px' => [
						'min' => 40,
						'max' => 1440,
					],
				],
				'selectors' => [
					'{{WRAPPER}} iframe' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'prevent_scroll',
			[
				'label' => __( 'Prevent Scroll', 'royarch' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'selectors' => [
					'{{WRAPPER}} iframe' => 'pointer-events: none;',
				],
			]
		);

		$this->add_control(
			'view',
			[
				'label' => __( 'View', 'royarch' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render Royarch_Elementor_Global_Widgets_Contact_Block widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render() {
        $settings = $this->get_settings_for_display();?>

		<!-- contact-info-section 
			================================================== -->
		<section class="contact-info-section">

			<div class="container">
				<div class="title-box">
					<span><?php echo do_shortcode($settings['widget_sub']); ?></span>
					<h2><?php echo do_shortcode($settings['widget_title']); ?></h2>
				</div>
				<div class="contact-info-box">
					<div class="row g-0">
						<?php foreach (  $settings['list'] as $item ) { ?>
							<div class="col-lg-4">
								<div class="contact-info-post">
									<?php
									if (strpos($item['list_icon']['value'],'fa-') !== false) {
										Icons_Manager::render_icon( $item['list_icon'], [ 'aria-hidden' => 'true' ] );
									} elseif (strpos($item['list_icon']['value'],'material-icons-outlined') !== false) {
										?>
										<span class="<?php echo esc_attr($item['list_icon']['value']); ?>">
										<?php 
											echo esc_html(str_replace( 'material-icons-outlined md-', '', $item['list_icon']['value']  ));
										?>
										</span>
										<?php
									} else { 
										?>
										<span class="<?php echo esc_attr($item['list_icon']['value']); ?>">
										<?php 
											echo esc_html(str_replace( 'material-icons md-', '', $item['list_icon']['value']  ));
										?>
										</span>
										<?php 
									}
									?>
									<div class="info-content">
										<h4><?php echo esc_html($item['list_title']); ?></h4>
										<p><?php echo esc_html($item['list_desc']); ?> </p>
									</div>
								</div>
							</div>
						<?php } ?>

					</div>
				</div>
				<div class="contact-form-box">
					<div class="row g-0">
						<div class="col-lg-8">
							<div id="contact-form">
								<?php echo do_shortcode($settings['widget_contact']); ?>
							</div>
						</div>
						<div class="col-lg-4">
							<div id="map">
								<?php
									if ( empty( $settings['address'] ) ) {
										return;
									}

									if ( 0 === absint( $settings['zoom']['size'] ) ) {
										$settings['zoom']['size'] = 10;
									}

									printf(
										'<div class="elementor-custom-embed"><iframe frameborder="0" scrolling="no" marginheight="0" marginwidth="0" src="https://maps.google.com/maps?q=%s&amp;t=m&amp;z=%d&amp;output=embed&amp;iwloc=near"></iframe></div>',
										urlencode( $settings['address'] ),
										absint( $settings['zoom']['size'] )
									);
								?>
							</div>
						</div>
					</div>
				</div>
			</div>

		</section>
		<!-- End contact-info-section -->
			
    
    <?php

	}
}
